package com.saucelabs.visual;

import com.saucelabs.visual.exception.VisualApiException;
import com.saucelabs.visual.graphql.type.BuildIn;
import com.saucelabs.visual.graphql.type.CreateSnapshotFromWebDriverIn;
import com.saucelabs.visual.graphql.type.FinishBuildIn;
import com.saucelabs.visual.graphql.type.RegionIn;
import com.saucelabs.visual.graphql.type.WebdriverSessionInfoIn;
import com.saucelabs.visual.graphqloperation.CreateBuildMutation;
import com.saucelabs.visual.graphqloperation.FinishBuildMutation;
import com.saucelabs.visual.graphqloperation.WebdriverSessionInfoQuery;
import com.saucelabs.visual.model.IgnoreRegion;
import com.saucelabs.visual.utils.ConsoleColors;
import com.saucelabs.visual.utils.GraphQLClient;

import org.openqa.selenium.Rectangle;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class VisualApi {
    private static final Logger log = LoggerFactory.getLogger(VisualApi.class);

    /**
     * Creates a VisualApi instance using builder style
     */
    public static class Builder {
        private final RemoteWebDriver driver;
        private final String username;
        private final String accessKey;
        private final String endpoint;
        private String projectName;
        private String branchName;
        private String buildName;

        public Builder(RemoteWebDriver driver, String username, String accessKey) {
            this(driver, username, accessKey, Region.US_WEST_1.endpoint);
        }

        public Builder(RemoteWebDriver driver, String username, String accessKey, Region region) {
            this(driver, username, accessKey, region.endpoint);
        }

        public Builder(RemoteWebDriver driver, String username, String accessKey, String endpoint) {
            this.driver = driver;
            this.username = username;
            this.accessKey = accessKey;
            this.endpoint = endpoint;
        }

        public Builder withBuild(String buildName) {
            this.buildName = buildName;
            return this;
        }

        public Builder withProject(String projectName) {
            this.projectName = projectName;
            return this;
        }

        public Builder withBranch(String branchName) {
            this.branchName = branchName;
            return this;
        }

        public VisualApi build() {
            if (this.buildName == null) {
                this.buildName = System.getenv("BUILD_NAME");
            }

            return new VisualApi(driver, endpoint, username, accessKey,
                    new BuildAttributes(buildName, projectName, branchName));
        }
    }

    private final GraphQLClient client;

    private final String sessionMetadataBlob;
    private final String buildId;
    private final String jobId;
    private final String sessionId;

    /**
     * Creates a VisualApi instance for a given Visual Backend {@link Region}
     *
     * @param driver    The {@link org.openqa.selenium.WebDriver} instance where the
     *                  tests should run at
     * @param region    Visual Backend Region. For available values, see:
     *                  {@link Region}
     * @param username  SauceLabs username
     * @param accessKey SauceLabs access key
     */
    public VisualApi(RemoteWebDriver driver, Region region, String username, String accessKey) {
        this(driver, region.endpoint, username, accessKey);
    }

    /**
     * Creates a VisualApi instance with a custom backend URL
     *
     * @param driver    The {@link org.openqa.selenium.WebDriver} instance where the
     *                  tests should run at
     * @param url       Visual Backend URL
     * @param username  SauceLabs username
     * @param accessKey SauceLabs access key
     */
    public VisualApi(RemoteWebDriver driver, String url, String username, String accessKey) {
        this(driver, url, username, accessKey, new BuildAttributes(null, null, null));
    }

    /**
     * Creates a VisualApi instance with a custom backend URL
     *
     * @param url             Visual Backend URL
     * @param username        SauceLabs username
     * @param accessKey       SauceLabs access key
     * @param buildAttributes like buildName, project, branch
     */
    public VisualApi(RemoteWebDriver driver, String url, String username, String accessKey,
            BuildAttributes buildAttributes) {
        if (username == null || accessKey == null || username.trim().isEmpty() || accessKey.trim().isEmpty()) {
            throw new VisualApiException("Invalid SauceLabs credentials. " +
                    "Please check your SauceLabs username and access key at https://app.saucelabs.com/user-settings");
        }
        this.client = new GraphQLClient(url, username, accessKey);
        this.sessionId = driver.getSessionId().toString();
        var jobIdString = (String) driver.getCapabilities().getCapability("jobUuid");
        this.jobId = jobIdString == null ? sessionId : jobIdString;
        this.buildId = VisualBuild.getOnce(this, buildAttributes);
        this.sessionMetadataBlob = this.webdriverSessionInfo().blob;
    }

    VisualApi(String jobId, String sessionId, String buildId, String sessionMetadataBlob, String url, String username, String accessKey) {
        if (username == null || accessKey == null || username.trim().isEmpty() || accessKey.trim().isEmpty()) {
            throw new VisualApiException("Invalid SauceLabs credentials. " +
                    "Please check your SauceLabs username and access key at https://app.saucelabs.com/user-settings");
        }
        this.buildId = buildId;
        this.jobId = jobId;
        this.sessionId = sessionId;
        this.client = new GraphQLClient(url, username, accessKey);
        this.sessionMetadataBlob = sessionMetadataBlob;
    }

    private WebdriverSessionInfoQuery.Result webdriverSessionInfo() {
        var input = WebdriverSessionInfoIn.builder().jobId(this.jobId).sessionId(this.sessionId).build();
        return this.client.execute(new WebdriverSessionInfoQuery(input)).result;
    }

    /**
     * Generates a string to give the build link.
     *
     * @param url The url to the VisualBuild
     */
    private static String getStartupMessage(String url) {
        StringBuilder sb = new StringBuilder();
        sb.append("\n")
                .append("\n")
                .append(ConsoleColors.Bold(ConsoleColors.Yellow("Sauce Visual\n")))
                .append("\n")
                .append(String.format("%100s\n", url))
                .append("\n");
        return sb.toString();
    }

    public static class BuildAttributes {
        private final String name;
        private final String project;
        private final String branch;

        public BuildAttributes(String name, String project, String branch) {
            this.name = name;
            this.project = project;
            this.branch = branch;
        }

        public String getName() {
            return name;
        }

        public String getProject() {
            return project;
        }

        public String getBranch() {
            return branch;
        }
    }

    String createBuild(String buildName) {
        return createBuild(new BuildAttributes(buildName, null, null));
    }

    String createBuild(BuildAttributes buildAttributes) {
        var input = BuildIn.builder()
                .name(buildAttributes.getName())
                .project(buildAttributes.getProject())
                .branch(buildAttributes.getBranch())
                .build();
        var mutation = new CreateBuildMutation(input);
        var data = client.execute(mutation);
        log.info(getStartupMessage(data.result.url));
        return data.result.id;
    }

    /**
     * Executes a finishBuild mutation
     *
     * @param buildId Build id
     * @return Finished build
     */
    FinishBuildMutation.Result finishBuild(String buildId) {
        var input = FinishBuildIn.builder().id(buildId).build();
        var mutation = new FinishBuildMutation(input);
        return client.execute(mutation).result;
    }

    /**
     * Uploads and creates a snapshot with a given name and default options
     *
     * @param name A name for the snapshot
     */
    public void check(String name) {
        Options options = new Options();
        check(name, options);
    }

    /**
     * Uploads and creates a snapshot with a given name and options
     *
     * @param name    A name for the snapshot
     * @param options Options for the API
     */
    public void check(String name, Options options) {
        var input = CreateSnapshotFromWebDriverIn.builder()
                .sessionId(this.sessionId)
                .name(name)
                .buildId(this.buildId)
                .ignoreRegions(extractIgnoreList(options))
                .sessionMetadata(this.sessionMetadataBlob)
                .build();
        client.check(input);
    }

    private List<RegionIn> extractIgnoreList(Options options) {
        if (options == null) {
            return Collections.emptyList();
        }
        List<RegionIn> result = new ArrayList<>();
        for (int i = 0; i < options.getIgnoreElements().size(); i++) {
            WebElement element = options.getIgnoreElements().get(i);
            if (validate(element) == null) {
                throw new VisualApiException("options.ignoreElement[" + i + "] does not exist (yet)");
            }
            result.add(toIgnoreIn(element));
        }
        for (int i = 0; i < options.getIgnoreRegions().size(); i++) {
            IgnoreRegion ignoreRegion = options.getIgnoreRegions().get(i);
            if (validate(ignoreRegion) == null) {
                throw new VisualApiException("options.ignoreRegion[" + i + "] is an invalid ignore region");
            }
            result.add(toIgnoreIn(ignoreRegion));
        }
        return result;
    }

    private RegionIn toIgnoreIn(WebElement element) {
        Rectangle r = element.getRect();
        return RegionIn.builder().x(r.getX()).y(r.getY()).width(r.getWidth()).height(r.getHeight()).build();
    }

    private RegionIn toIgnoreIn(IgnoreRegion r) {
        return RegionIn.builder().x(r.getX()).y(r.getY()).width(r.getWidth()).height(r.getHeight()).build();
    }

    private WebElement validate(WebElement element) {
        if (element == null || !element.isDisplayed() || element.getRect() == null) {
            return null;
        }
        return element;
    }

    private IgnoreRegion validate(IgnoreRegion region) {
        if (region == null || region.getHeight() == null || region.getWidth() == null ||
                region.getX() == null || region.getY() == null) {
            return null;
        }
        if (region.getHeight() <= 0 || region.getWidth() <= 0) {
            return null;
        }
        return region;
    }

    String getBuildId() {
        return buildId;
    }

}
