package com.saucelabs.visual.utils;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.Base64;

import com.apollographql.apollo3.api.CustomScalarAdapters;
import com.apollographql.apollo3.api.Operation;
import com.apollographql.apollo3.api.Operation.Data;
import com.apollographql.apollo3.api.Operations;
import com.apollographql.apollo3.api.json.BufferedSinkJsonWriter;
import com.saucelabs.visual.exception.VisualApiException;
import com.saucelabs.visual.graphql.type.*;
import com.saucelabs.visual.graphqloperation.CreateBuildMutation;
import com.saucelabs.visual.graphqloperation.CreateSnapshotFromWebDriverMutation;
import com.saucelabs.visual.graphqloperation.FinishBuildMutation;
import com.saucelabs.visual.graphqloperation.WebdriverSessionInfoQuery;
import okio.Buffer;

public class GraphQLClient {

    private final String uri;
    private final HttpClient client;
    private final String authentication;

    public GraphQLClient(String uri, String username, String accessKey) {
        this.uri = uri;
        this.client = HttpClient.newBuilder().build();
        this.authentication = Base64.getEncoder().encodeToString((username + ":" + accessKey).getBytes());
    }

    WebdriverSessionInfoQuery.Result webdriverSessionInfo(WebdriverSessionInfoIn input) {
        return this.execute(new WebdriverSessionInfoQuery(input)).result;
    }

    CreateBuildMutation.Result createBuild(BuildIn input) {
        var mutation = new CreateBuildMutation(input);
        var data = this.execute(mutation);
        // FIXME log somewhere else (al)
        // log.info(getStartupMessage(data.result.url));
        return data.result;
    }

    FinishBuildMutation.Result finishBuild(FinishBuildIn input) {
        var mutation = new FinishBuildMutation(input);
        var data = this.execute(mutation);
        return data.result;
    }

    public CreateSnapshotFromWebDriverMutation.Result check(CreateSnapshotFromWebDriverIn input) {
        var mutation = new CreateSnapshotFromWebDriverMutation(input);
        return this.execute(mutation).result;
    }

    public <D extends Data> D execute(Operation<D> operation) {
        try (Buffer buffer = new okio.Buffer()) {
            BufferedSinkJsonWriter jsonWriter = new BufferedSinkJsonWriter(buffer, "  ");

            Operations.composeJsonRequest(operation, jsonWriter, CustomScalarAdapters.Empty);

            var request = HttpRequest.newBuilder()
                    .uri(URI.create(this.uri))
                    .header("Content-Type", "application/json")
                    .header("Accept", "application/json")
                    .header("Authorization", "Basic " + this.authentication)
                    .POST(HttpRequest.BodyPublishers.ofString(buffer.readUtf8())).build();

            var response = client.send(request, BodyHandlers.ofString());
            if (response.statusCode() < 200 || response.statusCode() >= 300) {
                throw new VisualApiException("Unexpected status code: " + response.statusCode() + " " + response.body());
            }

            return Operations.parseJsonResponse(operation, response.body(), CustomScalarAdapters.Empty)
                    .dataAssertNoErrors();
        } catch (IOException | InterruptedException e) {
            throw new VisualApiException("Error while executing GraqpQL request", e);
        }
    }

}