package com.simplj.di.core;

import com.simplj.di.exceptions.SdfException;
import com.simplj.di.internal.TypeUtil;

import java.lang.reflect.*;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

public class DependencyResolver {

    DependencyResolver() {

    }

    private BiFunction<Type, String, Object> typeResolverF;
    private Function<String, Object> idResolverF;
    private Tri<Type, String, Map<String, Object>, Object> dynTypeResolverF;
    private BiFunction<String, Map<String, Object>, Object> dynIdResolverF;

    private boolean isSetupDone;
    private final Object lock = new Object();

    /**
     * Loads and initializes <i>DependencyResolver</i> with dependencies/constants
     * @throws SdfException - Throws SdfExceptions
     * @param config - dependency configuration to setup SDF
     */
    public void setup(DependencyResolverConfig config) throws SdfException {
        if (!isSetupDone) {
            synchronized (lock) {
                if (!isSetupDone) {
                    Class<?> loaderClass = findLoaderClass();
                    if (loaderClass == null) throw new SdfException("Could not initialize SDF! Library might be broken.");
                    Consumer<String> loggerF = config.getLoggerF();
                    if (loggerF == null) {
                        System.out.println("WARN: No logger function is set. Setting default log to console!");
                        loggerF = System.out::println;
                    }
                    Object obj;
                    try {
                        Constructor<?> cons = loaderClass.getDeclaredConstructors()[0];
                        cons.setAccessible(true);
                        obj = cons.newInstance(config.dependencyProviders(), config.basePackages(), config.properties(), config.profile(), loggerF);
                    } catch (InvocationTargetException | InstantiationException | IllegalAccessException e) {
                        if (e.getCause() instanceof SdfException) {
                            throw (SdfException) e.getCause();
                        } else {
                            throw new SdfException("Could not initialize SDF! Unable to load dependency classes.");
                        }
                    }
                    Method[] ms = loaderClass.getDeclaredMethods();
                    int count = 0;
                    for (int i = 0; i < ms.length && count < 4; i++) {
                        Method m = ms[i];
                        if (!Modifier.isStatic(m.getModifiers()) && Object.class.equals(m.getReturnType())) {
                            if (m.getParameters().length == 1) {
                                m.setAccessible(true);
                                idResolverF = s -> invoke(obj, m, s);
                                count++;
                            } else if (m.getParameters().length == 2) {
                                if (m.getParameters()[0].getType().isInterface()) {
                                    m.setAccessible(true);
                                    typeResolverF = (t, g) -> invoke(obj, m, t, g);
                                } else {
                                    m.setAccessible(true);
                                    dynIdResolverF = (s, p) -> invoke(obj, m, s, p);
                                }
                                count++;
                            } else if (m.getParameters().length == 3) {
                                m.setAccessible(true);
                                dynTypeResolverF = (t, g, p) -> invoke(obj, m, t, g, p);
                                count++;
                            }
                        }
                    }
                    if (typeResolverF == null || idResolverF == null || dynTypeResolverF == null || dynIdResolverF == null)
                        throw new SdfException("Could not initialize SDF! Resolver method not found.");
                    isSetupDone = true;
                }
            }
        }
    }

    public <T> T resolve(Class<T> clazz) throws SdfException {
        return resolve(clazz, null);
    }
    public <T> T resolve(Class<T> clazz, String tag) throws SdfException {
        if (typeResolverF == null) throw new SdfException("DependencyResolver is not configured!");
        Object obj = typeResolverF.apply(clazz, tag);
        return TypeUtil.typeCast(clazz, obj);
    }

    public <T> T resolve(TypeClass<T> type) throws SdfException {
        return resolve(type, null);
    }
    public <T> T resolve(TypeClass<T> type, String tag) throws SdfException {
        if (typeResolverF == null) throw new SdfException("DependencyResolver is not configured!");
        Object obj = typeResolverF.apply(type.getType(), tag);
        return TypeUtil.typeCast(type.getRawType(), obj);
    }

    public <T> T resolve(String id, Class<T> type) throws SdfException {
        if (idResolverF == null) throw new SdfException("DependencyResolver is not configured!");
        Object obj = idResolverF.apply(id);
        return TypeUtil.typeCast(type, obj);
    }
    public <T> T resolve(String id, TypeClass<T> type) throws SdfException {
        return resolve(id, type.getRawType());
    }

    public <T> T dynamicResolve(Class<T> clazz, Map<String, Object> rtArgs) throws SdfException {
        return dynamicResolve(clazz, null, rtArgs);
    }
    public <T> T dynamicResolve(Class<T> clazz, String tag, Map<String, Object> rtArgs) throws SdfException {
        if (dynTypeResolverF == null) throw new SdfException("DependencyResolver is not configured!");
        Object obj = dynTypeResolverF.apply(clazz, tag, rtArgs);
        return TypeUtil.typeCast(clazz, obj);
    }

    public <T> T dynamicResolve(TypeClass<T> type, Map<String, Object> rtArgs) throws SdfException {
        return dynamicResolve(type, null, rtArgs);
    }
    public <T> T dynamicResolve(TypeClass<T> type, String tag, Map<String, Object> rtArgs) throws SdfException {
        if (dynTypeResolverF == null) throw new SdfException("DependencyResolver is not configured!");
        Object obj = dynTypeResolverF.apply(type.getType(), tag, rtArgs);
        return TypeUtil.typeCast(type.getRawType(), obj);
    }

    public <T> T dynamicResolve(String id, Class<T> type, Map<String, Object> rtArgs) throws SdfException {
        if (dynIdResolverF == null) throw new SdfException("DependencyResolver is not configured!");
        Object obj = dynIdResolverF.apply(id, rtArgs);
        return TypeUtil.typeCast(type, obj);
    }

    public <T> T dynamicResolve(String id, TypeClass<T> type, Map<String, Object> rtArgs) throws SdfException {
        return dynamicResolve(id, type.getRawType(), rtArgs);
    }

    private static Class<?> findLoaderClass() {
        Class<?> clazz;
        try {
            String pkg = DependencyResolver.class.getPackage().getName();
            clazz = Class.forName(String.format("%s.%s.%s", pkg.substring(0, pkg.lastIndexOf('.')), "internal", "DependencyLoader"));
        } catch (ClassNotFoundException ex) {
            throw new SdfException("Could not initialize SDF! Unable to find loader class.");
        }
        return clazz;
    }

    private Object invoke(Object obj, Method m, Type type, String tag) {
        try {
            return m.invoke(obj, type, tag);
        } catch (InvocationTargetException | IllegalAccessException e) {
            if (e.getCause() instanceof SdfException) {
                throw (SdfException) e.getCause();
            } else {
                throw new SdfException("Could not resolve '" + type.getTypeName() + "'. Error: Unable to invoke resolver method!");
            }
        }
    }
    private Object invoke(Object obj, Method m, String id) {
        try {
            return m.invoke(obj, id);
        } catch (InvocationTargetException | IllegalAccessException e) {
            if (e.getCause() instanceof SdfException) {
                throw (SdfException) e.getCause();
            } else {
                throw new SdfException("Could not resolve '" + id + "'. Error: Unable to invoke resolver method!");
            }
        }
    }
    private Object invoke(Object obj, Method m, Type type, String tag, Map<String, Object> params) {
        try {
            return m.invoke(obj, type, tag, params);
        } catch (InvocationTargetException | IllegalAccessException e) {
            if (e.getCause() instanceof SdfException) {
                throw (SdfException) e.getCause();
            } else {
                throw new SdfException("Could not resolve '" + type.getTypeName() + "'. Error: Unable to invoke resolver method!");
            }
        }
    }
    private Object invoke(Object obj, Method m, String id, Map<String, Object> params) {
        try {
            return m.invoke(obj, id, params);
        } catch (InvocationTargetException | IllegalAccessException e) {
            if (e.getCause() instanceof SdfException) {
                throw (SdfException) e.getCause();
            } else {
                throw new SdfException("Could not resolve '" + id + "'. Error: Unable to invoke resolver method!");
            }
        }
    }
}
