package com.simplj.di.core;

import com.simplj.di.annotations.Constant;
import com.simplj.di.annotations.DependencyProvider;

import java.util.Arrays;
import java.util.Properties;
import java.util.function.Consumer;

public class DependencyResolverConfig {
    private final Object[] dependencyProviders;
    private final String[] basePackages;
    private final Properties properties;
    private final String profile;
    private final Consumer<String> loggerF;

    private DependencyResolverConfig(Object[] dependencyProviders, String[] basePackages, Properties properties, String profile, Consumer<String> loggerF) {
        this.dependencyProviders = dependencyProviders;
        this.basePackages = basePackages;
        this.properties = properties;
        this.profile = profile;
        this.loggerF = loggerF;
    }

    Object[] dependencyProviders() {
        return dependencyProviders;
    }

    String[] basePackages() {
        return basePackages;
    }

    Properties properties() {
        return properties;
    }

    public String profile() {
        return profile;
    }

    public Consumer<String> getLoggerF() {
        return loggerF;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Object[] providers;
        private String[] packages;
        private Properties props;
        private String profile;
        private Consumer<String> loggerF;

        /**
         * Register Classes that has <i>@DependencyProvider</i>/<i>@Constant</i> definitions
         * @param dependencyProviders - Class Instances that has <i>@DependencyProvider</i>/<i>@Constant</i> definitions
         * @return Same instance of <i>Builder</i>
         */
        public Builder setDependencyProviders(Object...dependencyProviders) {
            for (Object dp : dependencyProviders) {
                if (Arrays.stream(dp.getClass().getDeclaredMethods()).noneMatch(m -> m.isAnnotationPresent(DependencyProvider.class) || m.isAnnotationPresent(Constant.class))) {
                    throw new IllegalArgumentException("Invalid Dependency Provider '" + dp.getClass().getName() + "'! No method in this class is annotated with @DependencyProvider.");
                }
            }
            this.providers = dependencyProviders;
            return this;
        }

        /**
         * register packages where scan will be performed to load <i>@Dependency</i> classes
         * @param packages - base packages
         * @return Same instance of <i>Builder</i>
         */
        public Builder setBasePackages(String...packages) {
            this.packages = packages;
            return this;
        }

        /**
         * Set static/constant Properties
         * @param props - Properties
         * @return Same instance of <i>Builder</i>
         */
        public Builder setProperties(Properties props) {
            this.props = props;
            return this;
        }

        /**
         * Set profile to load the profile specific dependencies
         * @param profile - Profile Name
         * @return Same instance of <i>Builder</i>
         */
        public Builder setProfile(String profile) {
            boolean isEmpty = profile == null || profile.isEmpty();
            if (isEmpty) {
                System.out.println("WARNING: Profile passed as '' (empty). This is same as resolving dependencies WITHOUT profile!");
            } else {
                this.profile = profile;
            }
            return this;
        }

        /**
         * Set logging function
         * @param loggerF - logging function
         */
        public Builder setLoggerF(Consumer<String> loggerF) {
            this.loggerF = loggerF;
            return this;
        }

        public DependencyResolverConfig build() {
            return new DependencyResolverConfig(providers, packages, props, profile, loggerF);
        }
    }
}
