package com.simplj.di.internal;

import com.simplj.di.annotations.Dependency;
import com.simplj.di.annotations.DependencyProvider;
import com.simplj.di.exceptions.SdfException;

import java.util.*;
import java.util.stream.Collectors;

final class DependencyDef {
    private final String id;
    private final String name;
    private final TypeRef typeRef;
    private final DependencyInstantiator instantiator;
    private final boolean isSingleton;
    private final boolean isDefault;
    private final String fullyQualifiedName;
    private final Set<String> tags;

    private final boolean isConstant;
    private final boolean isProvider;
    private final boolean isDependency;

    private final Map<Integer, Argument> dependencies;
    private boolean isRuntimeProvided;

    static DependencyDef newDependency(TypeRef typeRef, DependencyInstantiator instantiator, String fullyQualifiedName, Dependency annotation) {
        Set<String> tags = Arrays.stream(annotation.tags()).filter(CommonUtil::isNotEmpty).collect(Collectors.toSet());
        return new DependencyDef(annotation.id(), typeRef.name(), typeRef, instantiator, annotation.singleton(), annotation.isDefault(), fullyQualifiedName, false, false, new HashMap<>(), tags);
    }

    static DependencyDef newDependencyProvider(TypeRef typeRef, DependencyInstantiator instantiator, String fullyQualifiedName, DependencyProvider annotation) {
        Set<String> tags = Arrays.stream(annotation.tags()).filter(CommonUtil::isNotEmpty).collect(Collectors.toSet());
        return new DependencyDef(annotation.id(), typeRef.name(), typeRef, instantiator, annotation.singleton(), annotation.isDefault(), fullyQualifiedName, false, true, new HashMap<>(), tags);
    }

    static DependencyDef newConstant(String constantId, TypeRef typeRef, DependencyInstantiator instantiator, String fullyQualifiedName, String[] tags) {
        Set<String> ts = tags == null ? Collections.emptySet() : Arrays.stream(tags).filter(CommonUtil::isNotEmpty).collect(Collectors.toSet());
        return new DependencyDef(constantId, constantId, typeRef, instantiator, false, false, fullyQualifiedName, true, false, new HashMap<>(), ts);
    }

    /*
    For constants id and type would be same.
     */
    private DependencyDef(String id, String name, TypeRef typeRef, DependencyInstantiator instantiator, boolean isSingleton, boolean isDefault, String fullyQualifiedName, boolean isConstant, boolean isProvider, Map<Integer, Argument> arguments, Set<String> tags) {
        this.id = id == null ? null : id.trim();
        this.name = name == null ? null : name.trim();
        this.typeRef = typeRef;
        this.instantiator = instantiator;
        this.isSingleton = isSingleton;
        this.isDefault = isDefault;
        this.fullyQualifiedName = fullyQualifiedName;
        this.tags = tags;
        this.isConstant = isConstant;
        this.isProvider = isProvider;
        this.isDependency = !isConstant && !isProvider;
        this.dependencies = arguments;
    }

    void addArgument(int idx, Argument arg) throws SdfException {
        this.dependencies.put(idx, arg);
        this.isRuntimeProvided = isRuntimeProvided || arg.isRuntimeProvided();
    }

    DependencyDef updateVTypes(Map<String, TypeRef> vTypeMap) {
        this.typeRef.updateVTypes(vTypeMap);
        DependencyDef res = new DependencyDef(id, typeRef.name(), typeRef, instantiator, isSingleton, isDefault, fullyQualifiedName, isConstant, isProvider, this.dependencies, tags);
        res.isRuntimeProvided = this.isRuntimeProvided;
        return res;
    }

    public String getId() {
        return id;
    }

    String getName() {
        return name;
    }

    TypeRef getTypeRef() {
        return typeRef;
    }

    DependencyInstantiator getInstantiator() {
        return instantiator;
    }

    boolean isSingleton() {
        return isSingleton;
    }

    boolean isDefault() {
        return isDefault;
    }

    String getFullyQualifiedName() {
        return fullyQualifiedName;
    }

    public Set<String> getTags() {
        return tags;
    }

    boolean matchesTag(String tag) {
        boolean emptyTag = CommonUtil.isEmpty(tag);
        return (emptyTag && tags.isEmpty()) || (!emptyTag && tags.contains(tag));
    }

    boolean isConstant() {
        return isConstant;
    }

    public boolean isProvider() {
        return isProvider;
    }

    public boolean isDependency() {
        return isDependency;
    }

    public boolean isRuntimeProvided() {
        return isRuntimeProvided;
    }

    Map<Integer, Argument> getDependencies() {
        return dependencies;
    }

    Object instantiate(Class<?> type) {
        return instantiator.instantiate(type);
    }

    Object instantiate() {
        return instantiator.instantiate();
    }

    Object instantiate(Map<String, Object> rtParams) {
        return instantiator.instantiate(rtParams);
    }

    @Override
    public String toString() {
        return "DD{" +
                (CommonUtil.isEmpty(id) ? "" : id + "::") +
                typeRef +
                ", " + fullyQualifiedName +
                (isSingleton ? ", singleton" : "") +
                (isDefault ? ", default" : "") +
                (isConstant ? ", constant" : "") +
                (isProvider ? ", provider" : "") +
                (isDependency ? ", dependency" : "") +
                (isRuntimeProvided ? ", RuntimeProvided" : "") +
                '}';
    }
}