package com.simplj.di.internal;

import com.simplj.di.exceptions.SdfException;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

abstract class DependencyInstantiator {
    private Object[] args;
    private final Map<Integer, ParameterMeta> rtArgsKeyIdxMap = new HashMap<>();

    abstract Class<?> type();
    abstract Object instantiate(Object[] args) throws SdfException;

    DependencyInstantiator setArgs(Object[] args) {
        this.args = args;
        //Identify and mark runtime provided args
        Object arg;
        ParameterMeta rtp;
        for (int i = 0, argsLength = args.length; i < argsLength; i++) {
            arg = args[i];
            if (arg instanceof ParameterMeta) {
                rtp = (ParameterMeta) arg;
                rtArgsKeyIdxMap.put(i, rtp);
            }
        }
        return this;
    }

    boolean isRuntimeProvided() {
        return !this.rtArgsKeyIdxMap.isEmpty();
    }

    //Internal Use (while init-ing)
    <T> T instantiate(Class<T> type) throws SdfException {
        Object o = instantiate(args);
        return TypeUtil.typeCast(type, o);
    }

    //Actual use (while resolving)
    Object instantiate() throws SdfException {
        return instantiate(args);
    }

    //Actual use (while dynamic resolving)
    Object instantiate(Map<String, Object> rtParams) throws SdfException {
        Object[] rtArgs = Arrays.copyOf(args, args.length);
        Object temp;
        ParameterMeta rtp;
        for (Map.Entry<Integer, ParameterMeta> entry : rtArgsKeyIdxMap.entrySet()) {
            rtp = entry.getValue();
            temp = rtParams.get(rtp.key());
            if (temp == null) {
                if (rtp.isNullable()) {
                    rtArgs[entry.getKey()] = null;
                } else {
                    throw new SdfException("Failed to instantiate '" + type() + "'! Reason: Could not find not-nullable dependency [" + rtp.key() + "]. Hint: Mark the parameter as `nullable=true` if optional.");
                }
            } else {
                rtArgs[entry.getKey()] = TypeUtil.typeCast(rtp.type(), temp);
            }
        }
        return instantiate(rtArgs);
    }
}

final class ConstructorInstantiator extends DependencyInstantiator {
    private final Constructor<?> constructor;

    ConstructorInstantiator(Constructor<?> constructor) {
        this.constructor = constructor;
    }

    @Override
    Class<?> type() {
        return constructor.getDeclaringClass();
    }

    @Override
    public Object instantiate(Object[] args) throws SdfException {
        try {
            return constructor.newInstance(args);
        } catch (InstantiationException | InvocationTargetException | IllegalAccessException | IllegalArgumentException e) {
            throw new SdfException("Failed to instantiate '" + type() + "' through constructor! Reason: " + e.getMessage(), e.getCause());
        }
    }
}

final class MethodInstantiator extends DependencyInstantiator {
    private final Method method;
    private final Object source;

    MethodInstantiator(Method method, Object instance) {
        this.method = method;
        this.source = instance;
    }

    @Override
    boolean isRuntimeProvided() {
        return false;
    }

    @Override
    Class<?> type() {
        return method.getReturnType();
    }

    @Override
    public Object instantiate(Object[] args) throws SdfException {
        try {
            if (!method.isAccessible()) {
                method.setAccessible(true);
            }
            if (method.getParameterCount() != (args == null ? 0 : args.length)) {
                throw new SdfException("Error instantiating " + type().getTypeName() + " from " + method.getName() + "!\n\tExpected " + Arrays.stream(method.getParameters()).map(p -> p.getType().getTypeName()).collect(Collectors.toList()) + "\n\tFound: " + Arrays.toString(args));
            }
            return method.invoke(source, args);
        } catch (IllegalAccessException | InvocationTargetException e) {
            throw new SdfException("Failed to instantiate '" + type() + "' through method '" + method.getName() + "'! Reason: " + e.getMessage(), e.getCause());
        }
    }
}

final class SingletonInstantiator extends DependencyInstantiator {
    private final ConstructorInstantiator cInit;
    private final MethodInstantiator mInit;
    private final boolean isConsInit;
    private volatile boolean isInitialized;
    private volatile Object obj;

    SingletonInstantiator(Constructor<?> constructor) {
        this(new ConstructorInstantiator(constructor), null, true);
    }

    SingletonInstantiator(Method method, Object instance) {
        this(null, new MethodInstantiator(method, instance), false);
    }

    private SingletonInstantiator(ConstructorInstantiator c, MethodInstantiator m, boolean isConstructor) {
        this.cInit = c;
        this.mInit = m;
        this.isConsInit = isConstructor;
    }

    @Override
    Class<?> type() {
        return isConsInit ? cInit.type() : mInit.type();
    }

    @Override
    Object instantiate(Object[] args) throws SdfException {
        if (!isInitialized) {
            synchronized (this) {
                if (!isInitialized) {
                    if (cInit != null) {
                        obj = cInit.instantiate(args);
                    } else if (mInit != null) {
                        obj = mInit.instantiate(args);
                    }
                    isInitialized = true;
                }
            }
        }
        return obj;
    }
}

final class ConstantInstantiator extends DependencyInstantiator {
    private final Object value;

    ConstantInstantiator(Object value) {
        this.value = value;
    }

    @Override
    Class<?> type() {
        return value.getClass();
    }

    @Override
    Object instantiate(Object[] args) throws SdfException {
        return value;
    }
}