package com.simplj.di.internal;

import com.simplj.di.exceptions.SdfException;

import java.util.function.Function;

abstract class PrimitiveType {
    private final Class<?> primitive;
    private final Class<?> wrapper;
    private final Function<Object, Object[]> wrapperArrF;
    private final Function<Object[], Object> primitiveArrF;

    private PrimitiveType(Class<?> primitive, Class<?> wrapper, Function<Object, Object[]> wrapperArrF, Function<Object[], Object> primitiveArrF) {
        this.primitive = primitive;
        this.wrapper = wrapper;
        this.wrapperArrF = wrapperArrF;
        this.primitiveArrF = primitiveArrF;
    }

    static PrimitiveType[] supportedPrimitiveTypes() {
        return new PrimitiveType[] {new BoolType(),
                new ByteType(), new CharType(), new DoubleType(),
                new FloatType(), new IntType(), new FloatType(),
                new IntType(), new LongType(), new ShortType()};
    }

    abstract Function<String, Object> stringToWrapperF();

    String primitiveName() {
        return primitive.getTypeName();
    }

    String wrapperName() {
        return wrapper.getTypeName();
    }

    Class<?> primitiveType() {
        return primitive;
    }

    Class<?> wrapperType() {
        return wrapper;
    }

    Function<Object, Object[]> wrapperArrF() {
        return wrapperArrF;
    }

    Function<Object[], Object> primitiveArrF() {
        return primitiveArrF;
    }

    static final class BoolType extends PrimitiveType {
        private BoolType() {
            super(boolean.class, Boolean.class, BoolType::toBooleanArray, BoolType::fromBoolArray);
        }

        private static Object[] toBooleanArray(Object source) {
            Object[] res;
            boolean[] arr = (boolean[]) source;
            res = new Boolean[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromBoolArray(Object[] src) {
            boolean[] arr = new boolean[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(boolean.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Boolean::parseBoolean;
        }
    }

    static final class ByteType extends PrimitiveType {
        private ByteType() {
            super(boolean.class, Byte.class, ByteType::toByteArray, ByteType::fromByteArray);
        }

        private static Object[] toByteArray(Object source) {
            Object[] res;
            byte[] arr = (byte[]) source;
            res = new Byte[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromByteArray(Object[] src) {
            byte[] arr = new byte[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(byte.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Byte::parseByte;
        }
    }

    static final class CharType extends PrimitiveType {
        private CharType() {
            super(boolean.class, Character.class, CharType::toCharArray, CharType::fromCharArray);
        }

        private static Object[] toCharArray(Object source) {
            Object[] res;
            char[] arr = (char[]) source;
            res = new Character[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromCharArray(Object[] src) {
            char[] arr = new char[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(char.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return this::stringToChar;
        }

        private Object stringToChar(String s) {
            char res;
            if (s == null) {
                res = '\0';
            } else if (s.length() == 1) {
                res = s.charAt(0);
            } else throw new SdfException("Cannot convert value '" + s + "' to type 'Character'!");
            return res;
        }
    }

    static final class DoubleType extends PrimitiveType {
        private DoubleType() {
            super(double.class, Double.class, DoubleType::toDoubleArray, DoubleType::fromDoubleArray);
        }

        private static Object[] toDoubleArray(Object source) {
            Object[] res;
            double[] arr = (double[]) source;
            res = new Double[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromDoubleArray(Object[] src) {
            double[] arr = new double[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(double.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Double::parseDouble;
        }
    }

    static final class FloatType extends PrimitiveType {
        private FloatType() {
            super(float.class, Float.class, FloatType::toFloatArray, FloatType::fromFloatArray);
        }

        private static Object[] toFloatArray(Object source) {
            Object[] res;
            float[] arr = (float[]) source;
            res = new Float[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromFloatArray(Object[] src) {
            float[] arr = new float[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(float.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Float::parseFloat;
        }
    }

    static final class IntType extends PrimitiveType {
        private IntType() {
            super(int.class, Integer.class, IntType::toIntegerArray, IntType::fromIntArray);
        }

        private static Object[] toIntegerArray(Object source) {
            Object[] res;
            int[] arr = (int[]) source;
            res = new Integer[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromIntArray(Object[] src) {
            int[] arr = new int[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(int.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Integer::parseInt;
        }
    }

    static final class LongType extends PrimitiveType {
        private LongType() {
            super(long.class, Long.class, LongType::toLongArray, LongType::fromLongArray);
        }

        private static Object[] toLongArray(Object source) {
            Object[] res;
            long[] arr = (long[]) source;
            res = new Long[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromLongArray(Object[] src) {
            long[] arr = new long[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(long.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Long::parseLong;
        }
    }

    static final class ShortType extends PrimitiveType {
        private ShortType() {
            super(boolean.class, Short.class, ShortType::toShortArray, ShortType::fromShortArray);
        }

        private static Object[] toShortArray(Object source) {
            Object[] res;
            short[] arr = (short[]) source;
            res = new Short[arr.length];
            for (int i = 0; i < arr.length; i++) {
                res[i] = arr[i];
            }
            return res;
        }

        private static Object fromShortArray(Object[] src) {
            short[] arr = new short[src.length];
            for (int i = 0; i < src.length; i++) {
                arr[i] = TypeUtil.typeCast(short.class, src[i]);
            }
            return arr;
        }

        @Override
        Function<String, Object> stringToWrapperF() {
            return Short::parseShort;
        }
    }
}
