package com.simplj.di.internal;

import java.lang.reflect.*;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

abstract class TypeRef {
    boolean sn = false;
    private final Type rawType;

    protected TypeRef(Type rawType) {
        this.rawType = rawType;
    }

    boolean isAssignableTo(TypeRef ref) {
        return ref != null && ref.isAssignableFrom(this);
    }
    abstract boolean isAssignableFrom(TypeRef ref);
    abstract List<TypeRef> parents();
    abstract Set<String> vTypes();
    abstract void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator);
    abstract void updateVTypes(Map<String, TypeRef> varTypeMap);
    abstract boolean isTyped();
    abstract Kind kind();

    /**
     * Type name - includes generics/type variable/bounds type information as well in the name
     * @return type name along with the generics/type variable/bounds type information (if present)
     */
    abstract String name();
    /**
     * Type name - includes generics type information only (does not include type variable/bounds type information)
     * name and rawName same for CType/AType/WType/VType.
     * Different in GAType/PType (does not contain parametric type information) and BType/UBType/LBType (does not contain bounded type information)
     * @return type name with generics type information (without type variable/bounds type information)
     */
    abstract String rawName();
    /**
     * Typed name - returns the substituted type name for PType/GAType/BType/UBType/LBType
     * value of this for `PType/GAType/BType/UBType/LBType` are different from name() value
     * @return substituted type name
     */
    abstract String typedName();

    Type rawType() {
        return this.rawType;
    }

    CType getAsCType() {
        return (this instanceof CType) ? (CType) this : null;
    }
    AType getAsAType() {
        return (this instanceof AType) ? (AType) this : null;
    }
    PType getAsPType() {
        return (this instanceof PType) ? (PType) this : null;
    }
    GAType getAsGAType() {
        return (this instanceof GAType) ? (GAType) this : null;
    }
    BType getAsBType() {
        return (this instanceof BType) ? (BType) this : null;
    }
    UBType getAsUBType() {
        return (this instanceof UBType) ? (UBType) this : null;
    }
    LBType getAsLBType() {
        return (this instanceof LBType) ? (LBType) this : null;
    }
    WType getAsWType() {
        return (this instanceof WType) ? (WType) this : null;
    }
    VType getAsVType() {
        return (this instanceof VType) ? (VType) this : null;
    }

    boolean isStringClass() {
        return Optional.ofNullable(getAsCType()).map(c -> String.class.equals(c.getType())).orElse(false);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + " " + name();
    }

    static TypeRef fromClass(Class<?> clazz) {
        TypeRef res;
        Set<String> visitedTypes = new HashSet<>();
        List<TypeRef> parents = new LinkedList<>();
        if (CommonUtil.isEmpty(clazz.getTypeParameters())) {
            res = fromType(clazz, clazz, visitedTypes, true);
        } else {
            crawlParents(parents, clazz, clazz, visitedTypes);
            res = pType(clazz, clazz.getTypeParameters(), parents, visitedTypes);
        }
        return res;
    }

    static TypeRef fromMethod(Method method) {
        return fromType(method.getGenericReturnType(), method.getReturnType(), new HashSet<>(), true);
    }

    static TypeRef fromParameter(Parameter parameter) {
        return fromType(parameter.getParameterizedType(), parameter.getType(), new HashSet<>(), false);
    }

    static TypeRef fromType(Type type) {
        return fromType(type, null, new HashSet<>(), false);
    }

    static TypeRef fromType(Type type, Class<?> clazz, Set<String> visitedTypes, boolean crawlParents) {
        TypeRef res = null;
        if (type == null) return res;
        List<TypeRef> parents;
        if (type instanceof Class) {
            Class<?> c = (Class<?>) type;
            if (c.isArray()) {
                res = new AType(c, crawlParents);
            } /*else if (!Model.isEmpty(c.getTypeParameters())) {
                parents = new LinkedList<>();
                crawlParents(parents, clazz == null ? TypeUtil.toClass(type) : clazz, type, visitedTypes);
                res = pType(c, c.getTypeParameters(), parents, visitedTypes);
            } */else {
                if (crawlParents) {
                    parents = new LinkedList<>();
                    crawlParents(parents, clazz == null ? TypeUtil.toClass(type) : clazz, type, visitedTypes);
                } else {
                    parents = Collections.emptyList();
                }
                res = new CType(c, parents);
            }
        } else if (type instanceof ParameterizedType) {
            ParameterizedType pt = (ParameterizedType) type;
            Class<?> pClass;
            if (clazz == null) {
                pClass = TypeUtil.toClass(pt.getRawType());
            } else {
                pClass = clazz;
            }
            if (!visitedTypes.contains(pClass.getTypeName())) {
                if (crawlParents) {
                    parents = new LinkedList<>();
                    crawlParents(parents, pClass, type, visitedTypes);
                } else {
                    parents = Collections.emptyList();
                }
                res = pType(pClass, pt.getActualTypeArguments(), parents, visitedTypes);
            }
        } else if (type instanceof TypeVariable<?>) {
            TypeVariable<?> tv = (TypeVariable<?>) type;
            Type[] bounds = tv.getBounds();
            if (isEmpty(bounds)) {
                res = new VType(type, tv.getName());
            } else {
                List<TypeRef> typeBounds = getTypeRefs(clazz, bounds, visitedTypes);
                res = CommonUtil.isEmpty(typeBounds) ? new VType(type, tv.getName()) : new BType(type, tv.getName(), typeBounds);
            }
        } else if (type instanceof WildcardType) {
            WildcardType wt = (WildcardType) type;
            boolean noUpperBound = isEmpty(wt.getUpperBounds());
            boolean noLowerBound = isEmpty(wt.getLowerBounds());
            if (noUpperBound && noLowerBound) {
                res = new WType(type);
            } else if (noLowerBound) {
                res = new UBType(type, getTypeRefs(clazz, wt.getUpperBounds(), visitedTypes));
            } else {
                res = new LBType(type, getTypeRefs(clazz, wt.getLowerBounds(), visitedTypes));
            }
        } else if (type instanceof GenericArrayType) {
            GenericArrayType gat = (GenericArrayType) type;
            res = new GAType(gat);
        } else {
            System.out.println("Unhandled Type: " + type.getClass());
        }
        return res;
    }

    static void crawlParents(List<TypeRef> parents, Class<?> clazz, Type type, Set<String> visitedTypes) {
        if (clazz == null || clazz.equals(Object.class) || visitedTypes.contains(clazz.getTypeName())) return;
        visitedTypes.add(clazz.getTypeName());
        Map<String, TypeRef> varTypeMap = new HashMap<>();
        List<TypeRef> gTypes = null;
        if (type instanceof ParameterizedType) {
            gTypes = getTypeRefs(clazz, ((ParameterizedType) type).getActualTypeArguments(), visitedTypes);
            mapActualTypes(clazz, gTypes, varTypeMap);
        }
        Class<?> curr = clazz;
        Class<?> pClass = curr.getSuperclass();
        Type gType;
        ParameterizedType pType;
        boolean reachedTop = pClass == null || pClass.getName().equals("java.lang.Object");
        if (reachedTop) {
            crawlInterfaces(clazz, parents, curr, varTypeMap, visitedTypes);
        }
        while (!reachedTop) {
            gType = curr.getGenericSuperclass();
            if (gType instanceof ParameterizedType) {
                pType = (ParameterizedType) gType;
                //mapActualTypes(curr, gTypes, varTypeMap);
                gTypes = reMapTypes(clazz, curr.getSuperclass(), pType, varTypeMap, visitedTypes);
                parents.add(new PType(pClass, pType, gTypes, Collections.emptyList()));
            } else {
                if (!(gType instanceof Class)) {
                    System.out.println("Not handled type: " + gType.getTypeName());
                }
                parents.add(new CType(pClass, Collections.emptyList()));
            }
            crawlInterfaces(clazz, parents, curr, varTypeMap, visitedTypes);
            varTypeMap.clear();
            mapActualTypes(pClass, gTypes, varTypeMap);
            curr = pClass;
            pClass = curr.getSuperclass();
            //TODO: Check if this can be optimized using memoization???
            visitedTypes.clear();
            reachedTop = pClass == null || pClass.getName().equals("java.lang.Object");
        }
    }

    private static PType pType(Class<?> clazz, Type[] types, List<TypeRef> parents, Set<String> visitedTypes) {
        List<TypeRef> pRefs = getTypeRefs(clazz, types, visitedTypes);
        return new PType(clazz, new PTypeRef(clazz, types, null), pRefs, parents);
    }

    private static boolean isEmpty(Type[] types) {
        return CommonUtil.isEmpty(types) || types[0].equals(java.lang.Object.class);
    }
    private static List<Class<?>> loadClasses(Type[] types) {
        List<Class<?>> classes = new ArrayList<>(types.length);
        for (Type t : types) {
            classes.add(TypeUtil.toClass(t));
        }
        return classes;
    }
    private static List<TypeRef>  getTypeRefs(Class<?> pClass, Type[] types, Set<String> visitedTypes) {
        List<TypeRef> pRefs = new ArrayList<>(types.length);
        TypeRef typeRef;
        int i = 0;
        for (Type t : types) {
            typeRef = fromType(t, null, visitedTypes, true);
            if (typeRef == null && pClass != null) {
                //For cases like `class A<T extends A<T>>` - refer 'test.self.bounded.types.Flow' in test code
                if (Arrays.stream(pClass.getTypeParameters()[i].getBounds()).anyMatch(b -> isSame(pClass, b))) {
                    typeRef = fromType(t, null, Collections.emptySet(), false);
                }
            }
            if (typeRef != null) {
                pRefs.add(typeRef);
            }
            i++;
        }
        return pRefs;
    }

    private static boolean isSame(Class<?> c, Type t) {
        boolean res = false;
        if (t instanceof ParameterizedType) {
            ParameterizedType pt = (ParameterizedType) t;
            res = pt.getRawType().equals(c);
        }
        return res;
    }

    //This is different than scanGenericSuperClass because here ic and gt refers to same interface but in the latter one, clazz and parentType is not same (refers child and parent respectively)
    //Because of the diff, in this method, types are mapped first correctly from genActTypeMap and then updated genActTypeMap (just the opposite from the other method)
    private static void crawlInterfaces(Class<?> source, List<TypeRef> parents, Class<?> clazz, Map<String, TypeRef> varTypeMap, Set<String> visitedTypes) {
        Class<?>[] iClasses = clazz.getInterfaces();
        Type[] igTypes = clazz.getGenericInterfaces();
        Class<?> iClass;
        Type iType;
        if (iClasses.length != igTypes.length) {
            System.out.println("Weird Case: interfaces.length and genericInterfaces.length does not match for class: " + clazz.getName());
        }
        List<TypeRef> gTypes;
        for (int i = 0; i < iClasses.length; i++) {
            iClass = iClasses[i];
            iType = igTypes[i];
            if (iType instanceof ParameterizedType) {
                ParameterizedType pType = (ParameterizedType) iType;
                gTypes = reMapTypes(source, iClass, pType, varTypeMap, visitedTypes);
                mapActualTypes(iClass, gTypes, varTypeMap);
                parents.add(new PType(iClass, pType, gTypes, Collections.emptyList()));
            } else {
                if (!(iType instanceof Class)) {
                    System.out.println("Not handled type: " + iType.getTypeName());
                }
                parents.add(new CType(iClass, Collections.emptyList()));
            }
            crawlInterfaces(source, parents, iClass, varTypeMap, visitedTypes);
        }
    }

    private static void mapActualTypes(Class<?> clazz, List<TypeRef> gTypes, Map<String, TypeRef> varTypeMap) {
        if (clazz.getTypeParameters().length == 0) return;
        if (gTypes == null) {
            System.out.println("Weird Case: Super class has type variable but child class does not have. Should not have come here, instead have caught by compiler itself! Erroneous Class: " + clazz.getName());
            return;
        }
        if (clazz.getTypeParameters().length != gTypes.size()) {
            System.out.println("Weird Case: Type Parameters size does not match with Generic Type Parameters for class: " + clazz.getName());
            System.out.println(clazz.getTypeParameters().length + " vs " + gTypes);
            return;
        }
        TypeVariable<? extends Class<?>> tv;
        for (int i = 0; i < clazz.getTypeParameters().length; i++) {
            tv = clazz.getTypeParameters()[i];
            varTypeMap.put(tv.getName(), gTypes.get(i));
        }
    }

    private static List<TypeRef> reMapTypes(Class<?> source, Class<?> pClass, ParameterizedType pType, Map<String, TypeRef> varTypeMap, Set<String> visitedTypes) {
        List<TypeRef> currTypes = getTypeRefs(pClass, pType.getActualTypeArguments(), visitedTypes);
        String var;
        TypeRef tr;
        for (int i = 0; i < currTypes.size(); i++) {
            tr = currTypes.get(i);
            var = getTypeVarName(tr);
            if (var != null) {
                tr = varTypeMap.get(var);
                if (tr == null) {
                    System.out.println("WARNING: Could not substitute Type Variable " + currTypes.get(i).rawName() + " in class: " + pType.getTypeName() + "! Source Class: " + source.getTypeName());
                } else {
                    currTypes.set(i, tr);
                }
            }
        }
        return currTypes;
    }

    private static String getTypeVarName(TypeRef typeRef) {
        String res;
        switch (typeRef.kind()) {
            case Bounded:
                res = typeRef.getAsBType().varName();
                break;
            case Variable:
                res = typeRef.name();
                break;
            default:
                res = null;
        }
        return res;
    }
}

enum Kind {
    /**
     * Concrete/Array/Bounded/UpperBounded/LowerBounded -> typedName()
     * Parameterized/GenericArray -> typedName()/rawName()
     */
    Concrete(0), Array(1), Parameterized(-1), GenericArray(-2), Bounded(2), UpperBounded(3), LowerBounded(4), Wildcard(5), Variable(6);
    private final int code;
    Kind(int code) {
        this.code = code;
    }
    public int code() {
        return code;
    }
}

//Concrete Types
final class CType extends TypeRef {
    private final Class<?> type;
    private final List<TypeRef> parentTypes;

    CType(Class<?> type, List<TypeRef> parents) {
        super(type);
        this.type = type;
        this.parentTypes = parents;
    }

    Class<?> getType() {
        return type;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        boolean res = false;
        if (ref != null) {
            switch (ref.kind()) {
                case Concrete:
                    CType cType = (CType) ref;
                    res = type.isAssignableFrom(cType.type);
                    break;
                case Parameterized:
                    PType pType = (PType) ref;
                    res = type.isAssignableFrom(pType.getType());
                    break;
                case Bounded:
                    BType bType = (BType) ref;
                    res = bType.getTypes().stream().allMatch(this::isAssignableFrom);
                    break;
                case UpperBounded:
                    UBType ubType = (UBType) ref;
                    res = ubType.getTypes().stream().allMatch(this::isAssignableFrom);
                    break;
                case LowerBounded:
                    LBType lbType = (LBType) ref;
                    res = lbType.getTypes().stream().allMatch(t -> t.isAssignableFrom(this));
                    break;
                case Variable:
                    VType vType = (VType) ref;
                    res = !vType.isTyped() || this.isAssignableFrom(vType.getType());
                    break;
            }
        }
        return res;
    }

    @Override
    public List<TypeRef> parents() {
        return parentTypes;
    }

    @Override
    Set<String> vTypes() {
        return Collections.emptySet();
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        //No VTypes for CType - Nothing to update!
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        //No VTypes for CType - Nothing to update!
    }

    @Override
    boolean isTyped() {
        return true;
    }

    @Override
    Kind kind() {
        return Kind.Concrete;
    }

    @Override
    String name() {
        return sn ? type.getSimpleName() : type.getTypeName();
    }

    @Override
    String rawName() {
        return name();
    }

    @Override
    String typedName() {
        return name();
    }
}

final class AType extends TypeRef {
    private final CType cType;
    private final int dimensions;
    private final String brackets;

    AType(Class<?> type, boolean crawlParents) {
        super(type);
        int t = 0;
        StringBuilder sb = new StringBuilder();
        while (type.isArray()) {
            t++;
            type = type.getComponentType();
            sb.append("[]");
        }
        List<TypeRef> parents;
        if (crawlParents) {
            parents = new LinkedList<>();
            crawlParents(parents, type, type, new HashSet<>());
        } else {
            parents = Collections.emptyList();
        }
        this.cType = new CType(type, parents);
        this.dimensions = t;
        this.brackets = sb.toString();
    }

    public CType getcType() {
        return cType;
    }

    public int getDimensions() {
        return dimensions;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        boolean res = false;
        if (ref != null) {
            switch (ref.kind()) {
                case Array:
                    AType aType = (AType) ref;
                    res = this.dimensions == aType.dimensions && this.cType.isAssignableFrom(aType.cType);
                    break;
                case GenericArray:
                    GAType gaType = (GAType) ref;
                    res = this.dimensions == gaType.getDimensions() && this.cType.isAssignableFrom(gaType.getType());
                    break;
            }
        }
        return res;
    }

    @Override
    public List<TypeRef> parents() {
        return cType.parents();
    }

    @Override
    Set<String> vTypes() {
        return Collections.emptySet();
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        //No VTypes for AType - Nothing to update!
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        //No VTypes for AType - Nothing to update!
    }

    @Override
    boolean isTyped() {
        return true;
    }

    @Override
    Kind kind() {
        return Kind.Array;
    }

    @Override
    String name() {
        return String.format("%s%s", cType.name(), brackets);
    }

    @Override
    String rawName() {
        return name();
    }

    @Override
    String typedName() {
        return name();
    }

    String formattedTypeName(TypeRef ref) {
        return String.format("%s%s", ref.typedName(), brackets);
    }
}

final class PType extends TypeRef {
    private final Class<?> type;
    private final List<TypeRef> pTypes;
    private final Set<String> vTypes;
    private final List<TypeRef> parentTypes;

    public PType(Class<?> clazz, Type rawType, List<TypeRef> pTypes, List<TypeRef> parents) {
        super(rawType);
        this.type = clazz;
        this.pTypes = pTypes;
        this.vTypes = pTypes.stream().flatMap(t -> t.vTypes().stream()).collect(Collectors.toSet());
        this.parentTypes = parents;
    }

    public Class<?> getType() {
        return type;
    }

    public List<TypeRef> getPTypes() {
        return pTypes;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        boolean res = false;
        if (ref != null) {
            switch (ref.kind()) {
                case Concrete:
                    CType cType = (CType) ref;
                    List<TypeRef> cParents = cType.parents();
                    for (int i = 0; !res && i < cParents.size(); i++) {
                        res = isAssignableFrom(cParents.get(i));
                    }
                    break;
                case Parameterized:
                    PType pType = (PType) ref;
                    if (type.isAssignableFrom(pType.getType())) {
                        res = pTypes.size() == pType.pTypes.size();
                        for (int i = 0; res && i < pTypes.size(); i++) {
                            res = pTypes.get(i).isAssignableFrom(pType.pTypes.get(i));
                        }
                        if (!res) {
                            List<TypeRef> pParents = pType.parentTypes;
                            for (int i = 0; !res && i < pParents.size(); i++) {
                                res = isAssignableFrom(pParents.get(i));
                            }
                        }
                    }
                    break;
                case Bounded:
                    BType bType = (BType) ref;
                    res = bType.getTypes().stream().allMatch(this::isAssignableFrom);
                    break;
                case UpperBounded:
                    UBType ubType = (UBType) ref;
                    res = ubType.getTypes().stream().allMatch(this::isAssignableFrom);
                    break;
                case LowerBounded:
                    LBType lbType = (LBType) ref;
                    res = lbType.getTypes().stream().allMatch(t -> t.isAssignableFrom(this));
                    break;
                case Variable:
                    VType vType = (VType) ref;
                    res = vType.isTyped() && this.isAssignableFrom(vType.getType());
                    break;
            }
        }
        return res;
    }

    @Override
    public List<TypeRef> parents() {
        return parentTypes;
    }

    @Override
    Set<String> vTypes() {
        return vTypes;
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        PType pType = ref.getAsPType();
        if (pType != null && this.pTypes.size() == pType.pTypes.size()) {
            for (int i = 0; i < this.pTypes.size(); i++) {
                this.pTypes.get(i).updateVTypesFrom(pType.pTypes.get(i), accumulator);
            }
            Map<String, TypeRef> parentMap = new HashMap<>();
            for (TypeRef p : pType.parentTypes) {
                parentMap.put(p.rawName(), p);
            }
            TypeRef temp;
            for (TypeRef p : this.parentTypes) {
                temp = parentMap.get(p.rawName());
                if (temp != null) {
                    p.updateVTypesFrom(temp, accumulator);
                }
            }
        }
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        for (TypeRef pType : this.pTypes) {
            pType.updateVTypes(varTypeMap);
        }
        for (TypeRef pt : this.parentTypes) {
            pt.updateVTypes(varTypeMap);
        }
    }

    @Override
    boolean isTyped() {
        return pTypes.stream().allMatch(TypeRef::isTyped);
    }

    @Override
    Kind kind() {
        return Kind.Parameterized;
    }

    @Override
    String name() {
        return String.format("%s<%s>", sn ? type.getSimpleName() : type.getTypeName(), pTypes.stream().map(TypeRef::name).collect(Collectors.joining(",")));
    }

    @Override
    String rawName() {
        return sn ? type.getSimpleName() : type.getTypeName();
    }

    @Override
    String typedName() {
        return String.format("%s<%s>", sn ? type.getSimpleName() : type.getTypeName(), pTypes.stream().map(TypeRef::typedName).collect(Collectors.joining(",")));
    }
}

final class GAType extends TypeRef {
    private final TypeRef type;
    private final int dimensions;
    private final String brackets;

    public GAType(GenericArrayType gaType) {
        super(gaType);
        int t = 1;
        StringBuilder sb = new StringBuilder("[]");
        Type type = gaType.getGenericComponentType();
        while (type instanceof GenericArrayType) {
            gaType = (GenericArrayType) type;
            t++;
            type = gaType.getGenericComponentType();
            sb.append("[]");
        }
        //TODO: Check here - whether this is ok or need to call the other fromType method
        this.type = fromType(type);
        this.dimensions = t;
        this.brackets = sb.toString();
    }

    public TypeRef getType() {
        return type;
    }

    public int getDimensions() {
        return dimensions;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        boolean res = false;
        if (ref != null) {
            switch (ref.kind()) {
                case Array:
                    AType aType = (AType) ref;
                    res = this.dimensions == aType.getDimensions() && this.type.isAssignableFrom(aType.getcType());
                    break;
                case GenericArray:
                    GAType gaType = (GAType) ref;
                    res = this.dimensions == gaType.dimensions && this.type.isAssignableFrom(gaType.type);
                    break;
            }
        }
        return res;
    }

    @Override
    public List<TypeRef> parents() {
        return type.parents();
    }

    @Override
    Set<String> vTypes() {
        return type.vTypes();
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        GAType gaType = ref.getAsGAType();
        if (gaType != null) {
            this.type.updateVTypesFrom(gaType.type, accumulator);
        }
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        this.type.updateVTypes(varTypeMap);
    }

    @Override
    boolean isTyped() {
        return type.isTyped();
    }

    @Override
    Kind kind() {
        return Kind.GenericArray;
    }

    @Override
    String name() {
        return String.format("%s%s", type.name(), brackets);
    }

    @Override
    String rawName() {
        return String.format("%s%s", type.rawName(), brackets);
    }

    @Override
    String typedName() {
        return String.format("%s%s", type.typedName(), brackets);
    }

    String formattedTypeName(TypeRef ref) {
        return String.format("%s%s", ref.typedName(), brackets);
    }

    String formattedRawName(TypeRef ref) {
        return String.format("%s%s", ref.rawName(), brackets);
    }
}

//Bounded Generic Types. Can be multiple when used as "<T extends ClassA & InterfaceB>"
class BType extends TypeRef {
    private final String name;
    private final List<TypeRef> types;
    private final List<TypeRef> parents;
    private final Set<String> vTypes;
    private String typedName;
    private String rawName;

    BType(Type rawType, String name, List<TypeRef> types) {
        super(rawType);
        this.name = name;
        this.types = types;
        this.parents = types.isEmpty() ? Collections.emptyList() : types.stream().flatMap(t -> t.parents().stream()).collect(Collectors.toList());
        this.vTypes = types.isEmpty() ? Collections.emptySet() : types.stream().flatMap(t -> t.vTypes().stream()).collect(Collectors.toSet());
        this.typedName = types.stream().map(TypeRef::typedName).collect(Collectors.joining(" & "));
        this.rawName = types.stream().map(TypeRef::rawName).collect(Collectors.joining(" & "));
    }

    List<TypeRef> getTypes() {
        return types;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        boolean res = false;
        if (ref != null) {
            switch (ref.kind()) {
                case Concrete:
                    CType cType = (CType) ref;
                    res = types.stream().allMatch(t -> t.isAssignableFrom(cType));
                    break;
                case Parameterized:
                    PType pType = (PType) ref;
                    res = types.stream().allMatch(t -> t.isAssignableFrom(pType));
                    break;
                case Bounded:
                    BType bType = (BType) ref;
                    res = types.size() == bType.types.size();
                    for (int i = 0; res && i < types.size(); i++) {
                        res = types.get(i).isAssignableFrom(bType.types.get(i));
                    }
                    break;
                case UpperBounded:
                    UBType ubType = (UBType) ref;
                    res = types.size() == ubType.getTypes().size();
                    for (int i = 0; res && i < types.size(); i++) {
                        res = types.get(i).isAssignableFrom(ubType.getTypes().get(i));
                    }
                    break;
                case Variable:
                    VType vType = (VType) ref;
                    res = vType.isTyped() && this.isAssignableFrom(vType.getType());
                    break;
                //Lower bounded ref would be false because <T extends Number> and <? super Integer> are not interchangeable
            }
        }
        return res;
    }

    @Override
    public List<TypeRef> parents() {
        return parents;
    }

    @Override
    Set<String> vTypes() {
        return vTypes;
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        BType bType = ref.getAsBType();
        if (bType != null && this.types.size() == bType.types.size()) {
            for (int i = 0; i < this.types.size(); i++) {
                this.types.get(i).updateVTypesFrom(bType.types.get(i), accumulator);
            }
        }
        this.typedName = types.stream().map(TypeRef::typedName).collect(Collectors.joining(" & "));
        this.rawName = types.stream().map(TypeRef::rawName).collect(Collectors.joining(" & "));
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        for (TypeRef type : this.types) {
            type.updateVTypes(varTypeMap);
        }
        this.typedName = types.stream().map(TypeRef::typedName).collect(Collectors.joining(" & "));
        this.rawName = types.stream().map(TypeRef::rawName).collect(Collectors.joining(" & "));
    }

    @Override
    boolean isTyped() {
        return types.stream().allMatch(TypeRef::isTyped);
    }

    @Override
    Kind kind() {
        return Kind.Bounded;
    }

    @Override
    String name() {
        return String.format("%s extends %s", name, typedName);
    }

    @Override
    String rawName() {
        return rawName;
    }

    @Override
    String typedName() {
        return typedName;
    }

    String varName() {
        return name;
    }
}

final class UBType extends BType {
    UBType(Type rawType, List<TypeRef> ubTypes) {
        super(rawType, "?", ubTypes);
    }

    @Override
    Kind kind() {
        return Kind.UpperBounded;
    }
}

final class LBType extends TypeRef {
    private final List<TypeRef> types;
    private final Set<String> vTypes;
    private final String typedName;
    private final String rawName;

    LBType(Type rawType, List<TypeRef> types) {
        super(rawType);
        this.types = types;
        this.vTypes = types.isEmpty() ? Collections.emptySet() : types.stream().flatMap(t -> t.vTypes().stream()).collect(Collectors.toSet());
        this.typedName = types.stream().map(TypeRef::typedName).collect(Collectors.joining(" & "));
        this.rawName = types.stream().map(TypeRef::rawName).collect(Collectors.joining(" & "));
    }

    List<TypeRef> getTypes() {
        return types;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        boolean res = false;
        if (ref != null) {
            switch (ref.kind()) {
                case Concrete:
                    CType cType = (CType) ref;
                    res = types.stream().allMatch(cType::isAssignableFrom);
                    break;
                case Parameterized:
                    PType pType = (PType) ref;
                    res = types.stream().allMatch(pType::isAssignableFrom);
                    break;
                case LowerBounded:
                    LBType lbType = (LBType) ref;
                    res = types.size() == lbType.getTypes().size();
                    for (int i = 0; res && i < types.size(); i++) {
                        res = lbType.getTypes().get(i).isAssignableFrom(types.get(i));
                    }
                    break;
                case Variable:
                    VType vType = (VType) ref;
                    res = vType.isTyped() && this.isAssignableFrom(vType.getType());
                    break;
                //Lower/ bounded ref would be false because <T extends Number> and <? super Integer> are not interchangeable
            }
        }
        return res;
    }

    @Override
    public List<TypeRef> parents() {
        return Collections.emptyList();
    }

    @Override
    Set<String> vTypes() {
        return vTypes;
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        LBType lbType = ref.getAsLBType();
        if (lbType != null && this.types.size() == lbType.types.size()) {
            for (int i = 0; i < this.types.size(); i++) {
                this.types.get(i).updateVTypesFrom(lbType.types.get(i), accumulator);
            }
        }
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        for (TypeRef type : this.types) {
            type.updateVTypes(varTypeMap);
        }
    }

    @Override
    boolean isTyped() {
        return types.stream().allMatch(TypeRef::isTyped);
    }

    @Override
    Kind kind() {
        return Kind.LowerBounded;
    }

    @Override
    String name() {
        return String.format("? super %s", typedName);
    }

    @Override
    String rawName() {
        return rawName;
    }

    @Override
    String typedName() {
        return typedName;
    }
}

final class WType extends TypeRef {
    WType(Type rawType) {
        super(rawType);
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        return true;
    }

    @Override
    public List<TypeRef> parents() {
        return Collections.emptyList();
    }

    @Override
    Set<String> vTypes() {
        return Collections.emptySet();
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        //No VTypes for WType - Nothing to update!
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        //No VTypes for WType - Nothing to update!
    }

    @Override
    boolean isTyped() {
        return true;
    }

    @Override
    Kind kind() {
        return Kind.Wildcard;
    }

    @Override
    String name() {
        return "?";
    }

    @Override
    String rawName() {
        return name();
    }

    @Override
    String typedName() {
        return name();
    }
}

final class VType extends TypeRef {
    private final String name;
    private final AtomicBoolean isTyped;
    private TypeRef type;

    VType(Type rawType, String name) {
        super(rawType);
        this.name = name;
        this.isTyped = new AtomicBoolean(false);
    }

    public String getName() {
        return name;
    }

    public TypeRef getType() {
        return type;
    }

    @Override
    boolean isAssignableFrom(TypeRef ref) {
        return !isTyped() || type.isAssignableFrom(ref);
    }

    @Override
    public List<TypeRef> parents() {
        return isTyped.get() ? type.parents() : Collections.emptyList();
    }

    @Override
    Set<String> vTypes() {
        return isTyped.get() ? type.vTypes() : Collections.singleton(name);
    }

    @Override
    void updateVTypesFrom(TypeRef ref, Map<String, TypeRef> accumulator) {
        boolean flag = isTyped.compareAndSet(false, true);
        if (flag) {
            this.type = ref;
            accumulator.put(name, type);
        }
    }

    @Override
    void updateVTypes(Map<String, TypeRef> varTypeMap) {
        TypeRef ref = varTypeMap.get(name);
        if (ref != null) {
            boolean flag = isTyped.compareAndSet(false, true);
            if (flag) {
                this.type = ref;
            }
        }
    }

    @Override
    public boolean isTyped() {
        return isTyped.get();
    }

    @Override
    Kind kind() {
        return Kind.Variable;
    }

    @Override
    String name() {
        return isTyped.get() ? type.name() : name;
    }

    @Override
    String rawName() {
        return isTyped.get() ? type.rawName() : name();
    }

    @Override
    String typedName() {
        return isTyped.get() ? type.typedName() : Object.class.getName();
    }
}