package com.simplj.di.internal;

import com.simplj.di.exceptions.SdfException;

import java.lang.reflect.Array;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

public final class TypeUtil {
    static final Map<String, PrimitiveType> primitiveTypeMap;
    static final Map<Class<?>, Function<String, Object>> wrapperFromStringMap;
    static final Set<String> primitiveTypes;
    static final Set<String> wrapperTypes;

    static {
        primitiveTypeMap = new HashMap<>();
        wrapperFromStringMap = new HashMap<>();
        PrimitiveType[] supportedPrimTypes = PrimitiveType.supportedPrimitiveTypes();
        for (PrimitiveType pt : supportedPrimTypes) {
            primitiveTypeMap.put(pt.primitiveName(), pt);
            wrapperFromStringMap.put(pt.wrapperType(), pt.stringToWrapperF());
        }
        primitiveTypes = primitiveTypeMap.keySet();
        wrapperTypes = primitiveTypeMap.values().stream().map(PrimitiveType::wrapperName).collect(Collectors.toSet());
    }

    public static Class<?> toClass(Type type) {
        try {
            return Class.forName(type.getTypeName());
        } catch (ClassNotFoundException e) {
            throw new SdfException("Could not resolve class: " + type.getTypeName() + "! Error: " + e.getMessage());
        }
    }
    /*
    Issues: Double -> Float,
     For Double.MAX -> Float getting Infinity
     For Double.MIN -> precision getting lost
     */
    public static <T> T typeCast(Class<T> type, Object o) throws SdfException {
        if (type == null) throw new SdfException("Type cannot be null!");
        if (o == null) {
            if (type.isPrimitive()) throw new SdfException("`null` cannot be set to primitive!");
            else return null;
        }
        if (!type.isAssignableFrom(o.getClass())) {
            if (o.getClass().equals(String.class) || wrapperTypes.contains(o.getClass().getTypeName())) {
                type = getWrapperIfPrimitives(type);
                o = convertFromString(o.toString(), type);
            } else if (type.isArray() && o.getClass().isArray()) {
                o = tryConvertingArrayTypes(type, o);
            } else {
                throw new SdfException("Unable get instance of type '" + type.getTypeName() + "' from type '" + o.getClass().getTypeName() + "'!");
            }
        }
        return type.cast(o);
    }

    static boolean isSupportedPrimitive(String name) {
        return primitiveTypes.contains(name);
    }

    static boolean isPrimitiveOrWrapper(String name) {
        return wrapperTypes.contains(name) || isSupportedPrimitive(name);
    }

    private static Object tryConvertingArrayTypes(Class<?> target, Object source) {
        Object res = null;
        if (source != null) {
            int sum = (target.isArray() ? 1 : 0) + (source.getClass().isArray() ? 1 : 0);
            //sum will be 2 if both are arrays or 0 if both are not. In other cases there should be error thrown!
            switch (sum) {
                case 2:
                    Object[] arr = toArray(source);
                    res = tryConvertingArrays(target.getComponentType(), arr);
                    break;
                case 0:
                    if (target.isAssignableFrom(source.getClass())) {
                        res = target.cast(source);
                    } else {
                        throw new SdfException("Could not convert arrays of type '" + source.getClass().getTypeName() + "' to type '" + target.getTypeName() + "'!");
                    }
                    break;
                default:
                    throw new SdfException("Could not convert arrays of type '" + source.getClass().getTypeName() + "' to type '" + target.getTypeName() + "'!");
            }
        }
        return res;
    }

    private static Object[] toArray(Object source) {
        Object[] res;
        PrimitiveType pt = primitiveTypeMap.get(source.getClass().getComponentType().getTypeName());
        if (pt == null) {
            res = (Object[]) source;
        } else {
            res = pt.wrapperArrF().apply(source);
        }
        return res;
    }

    private static <T> Object tryConvertingArrays(Class<T> type, Object[] src) {
        Object res;
        PrimitiveType pt = primitiveTypeMap.get(type.getTypeName());
        if (pt == null) {
            T[] arr = (T[]) Array.newInstance(type, src.length);
            for (int i = 0; i < src.length; i++) {
                arr[i] = typeCast(type, src[i]);
            }
            res = arr;
        } else {
            res = pt.primitiveArrF().apply(src);
        }
        return res;
    }

    private static <T> Class<T> getWrapperIfPrimitives(Class<T> type) {
        PrimitiveType pt = primitiveTypeMap.get(type.getTypeName());
        if (pt != null) {
            type = (Class<T>) pt.wrapperType();
        }
        return type;
    }

    private static Object convertFromString(String value, Class<?> to) {
        Object res;
        try {
            Function<String, Object> fromStringF = wrapperFromStringMap.get(to);
            if (fromStringF == null) {
                throw new SdfException("Cannot implicitly convert value '" + value + "' of type " + value.getClass().getTypeName() + " to type " + to.getTypeName() + '!');
            } else {
                res = fromStringF.apply(value);
            }
        } catch (Exception ex) {
            if (ex instanceof SdfException) throw ex;
            else throw new SdfException("Could not convert value '" + value + "' to type " + to.getTypeName());
        }
        return res;
    }
}
