package com.simplj.di.internal;

import com.simplj.di.annotations.Dependency;
import com.simplj.di.exceptions.SdfException;

import java.lang.reflect.Type;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

final class DependencyLoader {
    private final String SDF_CONST_KEY = "sdf_consts.";

    private final Class<?>[] dependencyProviders;
    private final String[] basePackages;
    private final Properties properties;
    private final String profile;
    private final String profileMessage;
    private final Consumer<String> logger;

    private final Map<String, Set<DependencyDef>> dependencyRefMap;

    private DependencyLoader(Class<?>[] dependencyProviders, String[] basePackages, Properties properties, String profile, Consumer<String> logger) throws SdfException {
        this.dependencyProviders = dependencyProviders == null ? new Class<?>[0] : dependencyProviders;
        this.basePackages = basePackages == null ? new String[0] : basePackages;
        this.properties = properties == null ? new Properties() : properties;
        if (profile == null) {
            this.profile = "";
            this.profileMessage = "";
        } else {
            this.profile = profile;
            this.profileMessage = String.format(" for profile '%s'", profile);
        }
        this.logger = logger;
        this.dependencyRefMap = new HashMap<>();
        init();
    }

    private Object get(Type type, String tag) throws SdfException {
        TypeRef typeRef = TypeRef.fromType(type);
        DependencyDef dd = DependencyUtil.getDependency(typeRef, tag, dependencyRefMap);
        if (dd == null) throw new SdfException("No Instance found for '" + typeRef.name() + '\'' + profileMessage + (CommonUtil.isEmpty(tag) ? "! Hint: Tag must be used to resolve tagged dependencies." : " with tag '" + tag + '\''));
        if (dd.isRuntimeProvided()) throw new SdfException(typeRef.rawName() + " has runtime provided dependencies and cannot be resolved through `resolve` method! Hint: Use `dynamicResolve()` with the necessary runtime parameters.");
        return dd.instantiate();
    }
    private Object get(Type type, String tag, Map<String, Object> rtParams) throws SdfException {
        TypeRef typeRef = TypeRef.fromType(type);
        DependencyDef di = DependencyUtil.getDependency(typeRef, tag, dependencyRefMap);
        if (di == null) throw new SdfException("No Instance found for '" + typeRef.name() + '\'' + profileMessage + (CommonUtil.isEmpty(tag) ? "! Hint: Tag must be used to resolve tagged dependencies." : " with tag '" + tag + '\''));
        return di.instantiate(rtParams);
    }
    private Object get(String id) throws SdfException {
        DependencyDef dd = DependencyUtil.getDependency(id, dependencyRefMap);
        if (dd == null) throw new SdfException("No Instance found for id '" + id + '\'' + profileMessage + '!');
        if (dd.isRuntimeProvided()) throw new SdfException(id + " has runtime provided dependencies and cannot be resolved through `resolve` method! Hint: Use `dynamicResolve()` with the necessary runtime parameters.");
        return dd.instantiate();
    }
    private Object get(String id, Map<String, Object> rtParams) throws SdfException {
        DependencyDef di = DependencyUtil.getDependency(id, dependencyRefMap);
        if (di == null) throw new SdfException("No Instance found for id '" + id + '\'' + profileMessage + '!');
        return di.instantiate(rtParams);
    }

    private void init() throws SdfException {
        long start = System.currentTimeMillis();
        logger.accept(" - SDF: Analyzing dependencies...");
        loadDependencies(dependencyRefMap);
        logger.accept(" - SDF: Analysis done. No CircularDependency or MissingDependency found! Instantiating...");
        Set<String> initializedDeps = new HashSet<>();
        int size = 0;
        for (Set<DependencyDef> ds : dependencyRefMap.values()) {
            for (DependencyDef d : ds) {
                if (!initializedDeps.contains(d.getName()) || !(CommonUtil.isEmpty(d.getId()) || initializedDeps.contains(d.getId()))) {
                    initialize(d, dependencyRefMap, initializedDeps);
                    size++;
                }
            }
        }
        logger.accept(" - SDF: Dependencies " + (profile.isEmpty() ? "" : "with Profile '" + profile + "' ") + "Instantiated Successfully!");
        //printMaps(3, null);
        long end = System.currentTimeMillis();
        logger.accept(" - SDF: Initialized with " + size + " dependencies/constants in " + formatTime(end - start));
    }

    private String formatTime(long millis) {
        return (millis / 1000.0) + " sec(s)";
    }

    private void printMaps(int n, String prefix) {
        if ((n & 1) == 1) {
            logger.accept("RefMap:");
            dependencyRefMap.forEach((k, v) -> {
                if (prefix == null || k.startsWith(prefix)) {
                    logger.accept(k + " ->\n\t" + v.stream().map(DependencyDef::toString).collect(Collectors.joining("\n\t")));
                }
            });
        }
    }

    private void loadDependencies(Map<String, Set<DependencyDef>> refMap) throws SdfException {
        loadConstants(refMap);
        List<DependencyDef> idBoundGenDeps = new LinkedList<>();
        Set<String> depDirectRefs = new HashSet<>();
        DependencyUtil.loadDependencyProviders(dependencyProviders, refMap, idBoundGenDeps, depDirectRefs, profile);
        for (String p : basePackages) {
            Set<Class<?>> deps = Scanner.scanForAnnotatedClasses(p, Dependency.class);
            DependencyUtil.loadDependencies(deps, refMap, idBoundGenDeps, depDirectRefs, profile);
        }
        DependencyUtil.enrichIdBoundGenDeps(idBoundGenDeps, refMap, depDirectRefs);
        DependencyUtil.validateLoadedDependencies(refMap, depDirectRefs);
    }

    private void loadConstants(Map<String, Set<DependencyDef>> depMap) throws SdfException {
        for (String k : System.getProperties().stringPropertyNames()) {
            if (k.startsWith(SDF_CONST_KEY)) {
                properties.setProperty(k.substring(SDF_CONST_KEY.length()), System.getProperty(k));
            }
        }
        DependencyUtil.loadConstantProviders(properties, dependencyProviders, depMap, profile);
    }

    private void initialize(DependencyDef d, Map<String, Set<DependencyDef>> refMap, Set<String> initializedDeps) throws SdfException {
        Collection<Argument> deps = d.getDependencies().values();
        Object[] args = new Object[deps.size()];
        Set<DependencyDef> subTypes;
        DependencyDef dep;
        for (Argument a : deps) {
            if (a.isRuntimeProvided()) {
                args[a.getIndex()] = a.getRuntimeProvided();
            } else if (a.isSubTypes()) {
                subTypes = DependencyUtil.getSubTypes(d, a.getTypeRef(), a.getTag(), refMap);
                for (DependencyDef st : subTypes) {
                    if (!initializedDeps.contains(st.getName()) || !(CommonUtil.isEmpty(st.getId()) || initializedDeps.contains(st.getId()))) {
                        initialize(st, refMap, initializedDeps);
                    }
                }
                args[a.getIndex()] = DependencyUtil.populateSubTypeArgs(a, subTypes);
            } else {
                dep = a.isIdBound() ? DependencyUtil.getDependency(a.getTypeNameOrId(), refMap) : DependencyUtil.getDependency(a.getTypeRef(), a.getTag(), refMap);
                if (!initializedDeps.contains(dep.getName()) || !(CommonUtil.isEmpty(dep.getId()) || initializedDeps.contains(dep.getId()))) {
                    initialize(dep, refMap, initializedDeps);
                }
                args[a.getIndex()] = dep.instantiate(a.getType());
            }
        }
        DependencyInstantiator dInit = d.getInstantiator();
        dInit.setArgs(args);
        initializedDeps.add(d.getName());
        if (!CommonUtil.isEmpty(d.getId())) {
            initializedDeps.add(d.getId());
        }
    }
}
