package com.simplj.di.internal;

import com.simplj.di.annotations.Constant;
import com.simplj.di.annotations.Dependency;
import com.simplj.di.annotations.DependencyProvider;
import com.simplj.di.exceptions.CircularDependencyException;
import com.simplj.di.exceptions.SdfException;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

final class DependencyUtil {
    static DependencyDef getDependency(TypeRef type, String tag, Map<String, Set<DependencyDef>> refMap) throws SdfException {
        DependencyDef res = null;
        Set<DependencyDef> deps = refMap.getOrDefault(type.typedName(), refMap.getOrDefault(type.rawName(), Collections.emptySet()))
                .stream().filter(d -> type.isAssignableFrom(d.getTypeRef()) && d.matchesTag(tag)).collect(Collectors.toSet());
        if (deps.size() == 1) {
            res = firstItem(deps);
        } else if (deps.size() > 1) {
            List<DependencyDef> defaults = deps.stream().filter(DependencyDef::isDefault).collect(Collectors.toList());
            if (defaults.isEmpty()) //multiple refs but no default
                throw new SdfException("Multiple implementations found for '" + type.name() + '\'' + (CommonUtil.isEmpty(tag) ? "" : " with tag '" + tag + '\'') + ": [" + deps.stream().map(DependencyDef::getName).collect(Collectors.joining(", ")) + "]");
            if (defaults.size() > 1) //multiple refs with default=true
                throw new SdfException("Multiple default implementations found for '" + type.name() + '\'' + (CommonUtil.isEmpty(tag) ? "" : " with tag '" + tag + '\'') + ": [" + defaults.stream().map(DependencyDef::getName).collect(Collectors.joining(", ")) + "]");
            res = defaults.get(0);
        }
        return res;
    }
    static DependencyDef getDependency(String id, Map<String, Set<DependencyDef>> refMap) throws SdfException {
        DependencyDef res = null;
        Set<DependencyDef> ds = refMap.get(id);
        if (ds != null && ds.size() == 1) {
            res = firstItem(ds);
        }
        return res;
    }

    static Set<DependencyDef> getSubTypes(DependencyDef self, TypeRef type, String tag, Map<String, Set<DependencyDef>> refMap) {
        return refMap.getOrDefault(type.rawName(), Collections.emptySet())
                .stream().filter(d -> type.isAssignableFrom(d.getTypeRef()) && !self.equals(d)).collect(Collectors.toSet());
    }

    static Object populateSubTypeArgs(Argument a, Set<DependencyDef> subTypes) throws SdfException {
        List<Object> resList = null;
        Map<String, Object> resMap = null;
        if (a.getSubTypesMeta().type().equals(Map.class)) {
            resMap = new HashMap<>();
        } else if (a.getSubTypesMeta().type().equals(List.class)) {
            resList = new LinkedList<>();
        } else {
            throw new SdfException("Only `Map` or `List` type is supported to have @SubTypes annotation!");
        }
        Object temp;
        for (DependencyDef d : subTypes) {
            temp = d.instantiate();
            if (temp != null) {
                if (resList != null) {
                    resList.add(temp);
                } else {
                    resMap.put(CommonUtil.isEmpty(d.getId()) ? d.getName() : d.getId(), temp);
                }
            }
        }
        return resList == null ? resMap : resList;
    }

    public static void enrichIdBoundGenDeps(List<DependencyDef> idBoundGenDeps, Map<String, Set<DependencyDef>> refMap, Set<String> depDirectRefs) {
        for (DependencyDef d : idBoundGenDeps) {
            enrichIdBoundGenDep(d, refMap, depDirectRefs);
        }
    }

    private static void enrichIdBoundGenDep(DependencyDef d, Map<String, Set<DependencyDef>> refMap, Set<String> depDirectRefs) {
        if (d.getTypeRef().isTyped()) return;
        Map<String, TypeRef> vTypeMap = new HashMap<>();
        for (Argument a : d.getDependencies().values()) {
            if (a.isIdBound() && !a.getTypeRef().vTypes().isEmpty()) {
                DependencyDef ad = getDependency(a.getTypeNameOrId(), refMap);
                if (ad == null) {
                    throw new SdfException("Could not load class " + d.getName() + ". Missing dependency '" + a.getTypeNameOrId() + (a.isSubstituted() ? "' - derived from " + a.getSubstitutedFrom() : "'") + (CommonUtil.isEmpty(a.getTag()) ? "! Hint: Tag must be used to resolve tagged dependencies." : " with tag '" + a.getTag() + '\''));
                }
                if (!ad.getTypeRef().isTyped()) {
                    enrichIdBoundGenDep(ad, refMap, depDirectRefs);
                }
                a.updateVTypesFrom(ad.getTypeRef(), vTypeMap);
            }
        }
        d = d.updateVTypes(vTypeMap);
        registerDependency(refMap, Collections.emptyList(), d.getTypeRef(), d, false, true, depDirectRefs);
    }

    static void validateLoadedDependencies(Map<String, Set<DependencyDef>> refMap, Set<String> depDirectRefs) throws SdfException {
        Set<String> validated = new HashSet<>();
        Set<String> path = new HashSet<>();
        //Validate Missing/Circular Dependency Check
        Iterator<Map.Entry<String, Set<DependencyDef>>> iter = refMap.entrySet().iterator();
        Map.Entry<String, Set<DependencyDef>> entry;
        Map<String, Map<String, List<String>>> defaultsMap = new HashMap<>();
        Map<String, Map<String, List<String>>> nonDefaultsMap = new HashMap<>();
        List<String> defaults = new LinkedList<>();
        List<String> nonDefaults = new LinkedList<>();
        StringBuilder errors = new StringBuilder();
        while (iter.hasNext()) {
            entry = iter.next();
            for (DependencyDef d : entry.getValue()) {
                validateDependency(d, refMap, path, validated);
                path.clear();
                if (depDirectRefs.contains(entry.getKey())) {
                    if (d.isDefault()) {
                        if (d.getTags().isEmpty()) {
                            defaults.add(d.getFullyQualifiedName());
                        }
                        for (String t : d.getTags()) {
                            defaultsMap.computeIfAbsent(t, x -> new HashMap<>()).computeIfAbsent(entry.getKey(), x -> new LinkedList<>()).add(d.getFullyQualifiedName());
                        }
                    } else {
                        if (d.getTags().isEmpty()) {
                            nonDefaults.add(d.getFullyQualifiedName());
                        }
                        for (String t : d.getTags()) {
                            nonDefaultsMap.computeIfAbsent(t, x -> new HashMap<>()).computeIfAbsent(entry.getKey(), x -> new LinkedList<>()).add(d.getFullyQualifiedName());
                        }
                    }
                }
            }
            if (defaults.size() > 1) {
                errors.append("\n\tMultiple implementations for '").append(entry.getKey()).append("' exists ").append(defaults).append(" and are marked as default");
            }
            if (defaults.size() == 0 && nonDefaults.size() > 1) {
                errors.append("\n\tMultiple implementations for '").append(entry.getKey()).append("' exists ").append(nonDefaults).append(" and none is marked as default");
            }
            nonDefaults.clear();
            defaults.clear();
        }
        defaultsMap.forEach((t, m) -> m.forEach((k, v) -> {
            if (v.size() > 1) {
                errors.append("\n\tMultiple implementations for '").append(k).append('\'').append(" with tag '").append(t).append("' exists ").append(defaults).append(" and are marked as default");
            }
        }));
        nonDefaultsMap.forEach((t, m) -> m.forEach((k, v) -> {
            if (v.size() > 1) {
                errors.append("\n\tMultiple implementations for '").append(k).append('\'').append(" with tag '").append(t).append("' exists ").append(nonDefaults).append(" and none is marked as default");
            }
        }));
        if (errors.length() > 0) {
            throw new SdfException("Found following multiple instances for same type!\n\tEven in case of multiple implementations with `id`, one must be marked as `default` so that no ambiguity is detected while resolving without id." + errors);
        }
    }

    private static void validateDependency(DependencyDef d, Map<String, Set<DependencyDef>> refMap, Set<String> path, Set<String> validated) throws SdfException {
        String className = d.getName();
        if (!validated.contains(className)) {
            if (path.contains(className)) throw new CircularDependencyException(className);
            Collection<Argument> dependencies = d.getDependencies().values();
            path.add(className);
            for (Argument a : dependencies) {
                if (!a.isRuntimeProvided()) {
                    if (a.isSubTypes()) {
                        validateSubTypeDependency(d, refMap, path, validated, className, a);
                    } else {
                        validateArgDependency(d, refMap, path, validated, className, a);
                    }
                } else if (d.isSingleton())
                    throw new SdfException("DynamicDependencies cannot be singleton! Parameter '" + a.getTypeRef().name() + "' in '" + d.getFullyQualifiedName() + "' is marked with @RuntimeProvided but the dependency is set as singleton. Hint: Use `singleton=false` in " + (d.isDependency() ? "@Dependency" : "@DependencyProvider") + " annotation for the dependency.");
            }
            validated.add(className);
            path.remove(className);
        }
    }

    private static void validateArgDependency(DependencyDef d, Map<String, Set<DependencyDef>> refMap, Set<String> path, Set<String> validated, String className, Argument a) {
        DependencyDef subD;
        subD = a.isIdBound() ? getDependency(a.getTypeNameOrId(), refMap) : getDependency(a.getTypeRef(), a.getTag(), refMap);
        if (subD == null)
            throw new SdfException("Could not load class " + className + ". Missing dependency '" + a.getTypeNameOrId() + (a.isSubstituted() ? " - derived from " + a.getSubstitutedFrom() : "'") + (CommonUtil.isEmpty(a.getTag()) ? "! Hint: Tag must be used to resolve tagged dependencies." : " with tag '" + a.getTag() + '\''));
        if (subD.isRuntimeProvided())
            throw new SdfException("RuntimeProvided dependencies cannot be used as a dependency! '" + d.getName() + "' has a runtime provided dependency '" + subD.getName() + "'.");
        if (!a.getTypeRef().isAssignableFrom(subD.getTypeRef()) && !(subD.getTypeRef().isStringClass() && TypeUtil.isPrimitiveOrWrapper(a.getTypeRef().name())))
            throw new SdfException("Type Mismatch! Type " + a.getTypeRef().name() + " at index " + a.getIndex() + " in class " + className + " does not match with loaded dependency " + subD.getTypeRef().name() + " for id or type: " + a.getTypeNameOrId());
        try {
            validateDependency(subD, refMap, path, validated);
        } catch (CircularDependencyException cd) {
            throw cd.addChain(className);
        }
    }

    private static void validateSubTypeDependency(DependencyDef parent, Map<String, Set<DependencyDef>> refMap, Set<String> path, Set<String> validated, String className, Argument a) {
        Set<DependencyDef> subTypes;
        subTypes = getSubTypes(parent, a.getTypeRef(), a.getTag(), refMap);
        if (subTypes == null) {
            if (a.getSubTypesMeta().isNullable()) return;
            else throw new SdfException("Could not load class " + className + ". No subtypes found for '" + a.getSubTypesMeta().key() + '\'');
        }
        for (DependencyDef d : subTypes) {
            if (!validated.contains(d.getName())) {
                try {
                    validateDependency(d, refMap, path, validated);
                } catch (CircularDependencyException cd) {
                    throw cd.addChain(className);
                }
            }
        }
    }

    static void loadConstantProviders(Properties properties, Class<?>[] classes, Map<String, Set<DependencyDef>> refMap, String profile) throws SdfException {
        Method[] methods;
        Constant annotation;
        Class<?> returnType;
        DependencyDef d;
        Set<DependencyDef> temp;
        String fqName;
        for (String k : properties.stringPropertyNames()) {
            if (isVariableKey(k))
                throw new SdfException("Invalid constant key: " + k + "! Constant keys cannot be variable.");
            d = DependencyDef.newConstant(k, TypeRef.fromClass(properties.getProperty(k).getClass()), new ConstantInstantiator(properties.getProperty(k)), k, null);
            refMap.put(k, Collections.singleton(d));
        }
        for (Class<?> c : classes) {
            methods = c.getDeclaredMethods();
            for (Method m : methods) {
                if (m.isAnnotationPresent(Constant.class)) {
                    if (!Modifier.isStatic(m.getModifiers()))
                        throw new SdfException("@Constant methods must be static. " + c.getName() + "." + m.getName() + " is not!");
                    annotation = m.getAnnotation(Constant.class);
                    if (matchesProfile(profile, annotation.profiles())) {
                        returnType = m.getReturnType();
                        if (!(returnType.getName().startsWith("java.lang") || TypeUtil.isSupportedPrimitive(returnType.getTypeName())))
                            throw new SdfException("@Constant cannot be used for type " + returnType.getName() + "! Please use @DependencyProvider instead.");
                        fqName = c.getName() + "." + m.getName();
                        if (isVariableKey(annotation.id()))
                            throw new SdfException("@Constant id cannot be variable key! Constant '" + fqName + "' has variable key in id.");
                        if (annotation.id().equals(returnType.getName()) || annotation.id().equals(returnType.getSimpleName()))
                            throw new SdfException("@Constant id cannot be same as it's type. Erroneous @Constant declaration: " + fqName);
                        d = DependencyDef.newConstant(annotation.id(), TypeRef.fromMethod(m), new SingletonInstantiator(m), fqName, annotation.tags());
                        //Check for existing value - if present throw error.
                        temp = refMap.get(annotation.id());
                        if (null != temp)
                            throw new SdfException("Multiple @Constant for id '" + annotation.id() + "' exists [" + d.getFullyQualifiedName() + ", " + firstItem(temp).getFullyQualifiedName() + "]");
                        identifyDependencies(d, m.getParameters(), refMap);
                        refMap.put(annotation.id(), Collections.singleton(d));
                    }
                }
            }
        }
    }

    static void loadDependencyProviders(Class<?>[] classes, Map<String, Set<DependencyDef>> refMap, List<DependencyDef> idBoundGenDeps, Set<String> depDirectRefs, String profile) throws SdfException {
        Method[] methods;
        DependencyProvider annotation;
        DependencyInstantiator instantiator;
        Class<?> returnType;
        TypeRef typeRef;
        String fqName;
        DependencyDef d;
        boolean idBoundGeneric;
        for (Class<?> c : classes) {
            methods = c.getDeclaredMethods();
            for (Method m : methods) {
                if (m.isAnnotationPresent(DependencyProvider.class)) {
                    if (!Modifier.isStatic(m.getModifiers()))
                        throw new SdfException("@DependencyProvider methods must be static. " + c.getName() + "." + m.getName() + " is not!");

                    annotation = m.getAnnotation(DependencyProvider.class);
                    if (matchesProfile(profile, annotation.profiles())) {
                        returnType = m.getReturnType();
                        if (returnType.getName().startsWith("java.lang"))
                            throw new SdfException("@DependencyProvider cannot be used for type " + returnType.getName() + "! Please use @Constant instead.");
                        instantiator = annotation.singleton() ? new SingletonInstantiator(m) : new MethodInstantiator(m);
                        fqName = c.getName() + "." + m.getName();
                        if (isVariableKey(annotation.id()))
                            throw new SdfException("@DependencyProvider id cannot be variable key! DependencyProvider '" + fqName + "' has variable key in id.");
                        if (annotation.id().equals(returnType.getName()) || annotation.id().equals(returnType.getSimpleName()))
                            throw new SdfException("@DependencyProvider id cannot be same as it's type. Erroneous @DependencyProvider declaration: " + fqName);
                        typeRef = TypeRef.fromMethod(m);
                        if (typeRef.kind().code() > 4)
                            throw new SdfException("@DependencyProvider cannot have TypeVariable or Wildcard as return type");
                        d = DependencyDef.newDependencyProvider(typeRef, instantiator, fqName, annotation);
                        idBoundGeneric = identifyDependencies(d, m.getParameters(), refMap);
                        registerDependency(refMap, idBoundGenDeps, typeRef, d, idBoundGeneric, false, depDirectRefs);
                    }
                }
            }
        }
    }

    static void loadDependencies(Set<Class<?>> classes, Map<String, Set<DependencyDef>> refMap, List<DependencyDef> idBoundGenDeps, Set<String> depDirectRefs, String profile) throws SdfException {
        TypeRef typeRef;
        Dependency annotation;
        DependencyInstantiator instantiator;
        Parameter[] params;
        Method method;
        Constructor<?> constructor;
        String fqName;
        DependencyDef d;
        boolean idBoundGeneric;
        for (Class<?> c : classes) {
            if (c.isAnnotationPresent(Dependency.class)) {
                if (c.isInterface() || Modifier.isAbstract(c.getModifiers()))
                    throw new SdfException("@Dependency annotation can only be used on a concrete class! '" + c.getName() + "' is an " + (c.isInterface() ? "Interface" : "Abstract Class"));
                typeRef = TypeRef.fromClass(c);
                if (typeRef.kind().code() > 4)
                    throw new SdfException("@Dependency cannot be a TypeVariable or Wildcard");
                annotation = c.getAnnotation(Dependency.class);
                if (matchesProfile(profile, annotation.profiles())) {
                    if (c.getConstructors().length != 1 && annotation.initMethod().isEmpty())
                        throw new SdfException("@Dependency classes must have only one public constructor or initMethod must be provided. " + c.getName() + " has " + c.getConstructors().length + " public constructor(s) and no initMethod provided!");
                    if (isVariableKey(annotation.id()))
                        throw new SdfException("@Dependency id cannot be variable key! Dependency '" + c.getName() + "' has variable key in id.");
                    if (!annotation.initMethod().isEmpty()) {
                        method = findMethod(c, annotation.initMethod());
                        if (!Modifier.isStatic(method.getModifiers()))
                            throw new SdfException("@Dependency initMethod must be static. " + c.getName() + "." + method.getName() + " is not!");
                        instantiator = annotation.singleton() ? new SingletonInstantiator(method) : new MethodInstantiator(method);
                        params = method.getParameters();
                        fqName = c.getName() + "." + method.getName();
                    } else {
                        constructor = c.getConstructors()[0];
                        fqName = constructorName(constructor);
                        if (!Modifier.isPublic(constructor.getModifiers())) {
                            throw new SdfException("Constructor " + fqName + " is not defined as `public` for class " + c.getName() + "!");
                        }
                        instantiator = annotation.singleton() ? new SingletonInstantiator(constructor) : new ConstructorInstantiator(constructor);
                        params = constructor.getParameters();
                    }
                    if (annotation.id().equals(c.getName()) || annotation.id().equals(c.getSimpleName()))
                        throw new SdfException("@Dependency id cannot be same as it's type. Erroneous @Dependency declaration: " + c.getName());
                    d = DependencyDef.newDependency(typeRef, instantiator, fqName, annotation);
                    idBoundGeneric = identifyDependencies(d, params, refMap);
                    registerDependency(refMap, idBoundGenDeps, typeRef, d, idBoundGeneric, false, depDirectRefs);
                }
            }
        }
    }

    //TODO: currently it is desired to have profiles for all implementations is profile name is used for the type.
    // Instead of this, need to have an option of having an implementation among the profile based implementations
    // which will not have any profile and this impl should be loaded only when configured without profile and this
    // impl should not be loaded when configured without profile
    private static boolean matchesProfile(String profile, String[] profiles) {
        boolean res = CommonUtil.isEmpty(profiles);
        for (int i = 0; !res && i < profiles.length; i++) {
            res = CommonUtil.isNotEmpty(profiles[i]) && profiles[i].equals(profile);
        }
        return res;
    }

    private static void registerDependency(Map<String, Set<DependencyDef>> refMap, List<DependencyDef> idBoundGenDeps, TypeRef typeRef, DependencyDef d, boolean idBoundGeneric, boolean fromEnrich, Set<String> depDirectRefs) {
        if (idBoundGeneric) {
            idBoundGenDeps.add(d);
        } else {
            loadParents(typeRef, refMap, d);
            refMap.computeIfAbsent(typeRef.typedName(), k -> new HashSet<>()).add(d);
            depDirectRefs.add(typeRef.typedName());
            refMap.computeIfAbsent(typeRef.rawName(), k -> new HashSet<>()).add(d);
        }
        if (!fromEnrich && !CommonUtil.isEmpty(d.getId())) {
            refMap.computeIfAbsent(d.getId(), k -> new HashSet<>()).add(d);
            depDirectRefs.add(d.getId());
        }
    }

    private static void loadParents(TypeRef typeRef, Map<String, Set<DependencyDef>> refMap, DependencyDef d) {
        Function<TypeRef, String> typedNameF;
        Function<TypeRef, String> rawNameF;
        switch (typeRef.kind()) {
            case Array:
                AType aType = typeRef.getAsAType();
                typedNameF = aType::formattedTypeName;
                rawNameF = TypeRef::rawName;
                break;
            case GenericArray:
                GAType gaType = typeRef.getAsGAType();
                typedNameF = gaType::formattedTypeName;
                rawNameF = gaType::formattedRawName;
                break;
            default:
                typedNameF = TypeRef::typedName;
                rawNameF = TypeRef::rawName;
                break;
        }
        for (TypeRef p : typeRef.parents()) {
            switch (p.kind()) {
                case Concrete:
                    parentNotSingletonDependency(typeRef, p.getAsCType().getType());
                    break;
                case Parameterized:
                    parentNotSingletonDependency(typeRef, p.getAsPType().getType());
                    break;
            }
            refMap.computeIfAbsent(typedNameF.apply(p), k -> new HashSet<>()).add(d);
            if (p.kind().code() < 0)
                refMap.computeIfAbsent(rawNameF.apply(p), k -> new HashSet<>()).add(d);
        }
    }

    private static void parentNotSingletonDependency(TypeRef type, Class<?> parentClass) {
        if (parentClass != null && parentClass.isAnnotationPresent(Dependency.class)) {
            Dependency annotation = parentClass.getAnnotation(Dependency.class);
            if (annotation.singleton()) {
                throw new SdfException("Singleton dependencies cannot be extended. Dependency '" + type.name() + "' extends singleton dependency: " + parentClass.getName() + "\nHint: Make the parent dependency non-singleton.");
            }
        }
    }

    private static Optional<String> excerptVar(String id, Function<String, Set<DependencyDef>> substitutorF) throws SdfException {
        Optional<String> res = Optional.empty();
        if (isVariableKey(id)) {
            String temp = id.substring(2, id.length() - 1);
            if (CommonUtil.isEmpty(temp)) throw new SdfException("Invalid variable key: " + id);
            Set<DependencyDef> deps = substitutorF.apply(temp);
            if (null == deps) throw new SdfException("Substitution Failed! No value found for variable key: " + id);
            if (deps.size() > 1) throw new SdfException("Substitution Failed! Multiple value found for variable key: " + id);
            DependencyDef dep = firstItem(deps);
            if (!dep.getTypeRef().isStringClass()) throw new SdfException("Substitution Failed! Value for a variable id must be of type 'java.lang.String'. " + id + " has type " + dep.getTypeRef().name());
            Object obj = dep.getInstantiator().instantiate();
            temp = obj instanceof String ? obj.toString() : null;
            if (null == temp) throw new SdfException("Variable id '" + id + "' substituted to `null`. a NonNull String value is expected here!");
            res = Optional.of(temp);
        }
        return res;
    }

    static boolean isVariableKey(String id) {
        return id != null && id.length() > 2 && '$' == id.charAt(0) && '{' == id.charAt(1) && '}' == id.charAt(id.length() - 1);
    }

    static <T> T firstItem(Set<T> ts) {
        return ts.iterator().next();
    }

    private static boolean identifyDependencies(DependencyDef d, Parameter[] params, Map<String, Set<DependencyDef>> refMap) throws SdfException {
        Argument arg;
        Set<String> vTypeArgs = new HashSet<>();
        Set<String> rtKeys = new HashSet<>();
        boolean idBoundGeneric = false;
        for (int i = 0; i < params.length; i++) {
            arg = new Argument(d.getFullyQualifiedName(), params[i], i, k -> excerptVar(k, refMap::get));
            idBoundGeneric = idBoundGeneric || (arg.isIdBound() && !arg.getTypeRef().vTypes().isEmpty());
            d.addArgument(i, arg);
            vTypeArgs.addAll(arg.getTypeRef().vTypes());
            if (arg.isRuntimeProvided() && !rtKeys.add(arg.getRuntimeProvided().key())) {
                throw new SdfException("@RuntimeProvided key must be unique in constructor/method! Key '" + arg.getRuntimeProvided().key() + "' is not unique in " + d.getFullyQualifiedName());
            }
        }
        Set<String> notMappedVTypes = d.getTypeRef().vTypes().stream().filter(v -> !vTypeArgs.contains(v)).collect(Collectors.toSet());
        if (!notMappedVTypes.isEmpty())
            throw new SdfException("TypeVariable(s) in a @Dependency/@DependencyProvider must either be bound using @Bind or marked as @RuntimeProvided for RuntimeProvided values! '" + d.getFullyQualifiedName() + "' has un-resolved TypeVariable(s): " + notMappedVTypes);
        return idBoundGeneric;
    }

    private static Method findMethod(Class<?> clazz, String methodName) throws SdfException {
        Method method = null;
        Method[] methods = clazz.getDeclaredMethods();
        for (int i = 0; i < methods.length; i++) {
            if (methodName.equals(methods[i].getName())) {
                method = methods[i];
                i = methods.length;
            }
        }
        if (method == null) throw new SdfException("Method " + methodName + " not found in class " + clazz.getName());
        return method;
    }

    private static String constructorName(Constructor<?> constructor) {
        return String.format("%s(%s)", constructor.getName(), Arrays.stream(constructor.getParameterTypes()).map(Class::getName).collect(Collectors.joining(",")));
    }
}
