// SPDX-License-Identifier: LGPL-2.1-or-later
// Copyright (c) 2012-2014 Monty Program Ab
// Copyright (c) 2015-2025 MariaDB Corporation Ab
// Copyright (c) 2021-2025 SingleStore, Inc.

package com.singlestore.jdbc;

import static com.singlestore.jdbc.util.constants.Capabilities.LOCAL_FILES;

import com.singlestore.jdbc.client.ColumnDecoder;
import com.singlestore.jdbc.client.result.CompleteResult;
import com.singlestore.jdbc.client.result.Result;
import com.singlestore.jdbc.client.util.ClosableLock;
import com.singlestore.jdbc.export.ExceptionFactory;
import com.singlestore.jdbc.message.ClientMessage;
import com.singlestore.jdbc.message.client.PreparePacket;
import com.singlestore.jdbc.message.client.QueryWithParametersPacket;
import com.singlestore.jdbc.message.client.RewriteQueryMultiPacket;
import com.singlestore.jdbc.message.server.OkPacket;
import com.singlestore.jdbc.util.ClientParser;
import com.singlestore.jdbc.util.ParameterList;
import com.singlestore.jdbc.util.RewriteClientParser;
import com.singlestore.jdbc.util.timeout.QueryTimeoutHandler;
import java.sql.BatchUpdateException;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLTimeoutException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;

public class ClientPreparedStatement extends BasePreparedStatement {

  private final ClientParser parser;

  /**
   * Client prepare statement constructor
   *
   * @param sql command
   * @param con connection
   * @param lock thread safe lock
   * @param autoGeneratedKeys must command return automatically generated keys
   * @param resultSetType resultset type
   * @param resultSetConcurrency resultset concurrency
   * @param defaultFetchSize default fetch size
   */
  public ClientPreparedStatement(
      String sql,
      Connection con,
      ClosableLock lock,
      int autoGeneratedKeys,
      int resultSetType,
      int resultSetConcurrency,
      int defaultFetchSize) {
    super(
        sql,
        con,
        lock,
        false,
        autoGeneratedKeys,
        resultSetType,
        resultSetConcurrency,
        defaultFetchSize);
    parser = ClientParser.parameterParts(sql);
    parameters = new ParameterList(parser.getParamCount());
  }

  @SuppressWarnings("try")
  private void executeInternal() throws SQLException {
    checkNotClosed();
    validParameters();
    try (ClosableLock ignore = lock.closeableLock();
        QueryTimeoutHandler ignore2 = this.con.handleTimeout(queryTimeout)) {
      QueryWithParametersPacket query =
          new QueryWithParametersPacket(null, parser, parameters, localInfileInputStream);
      results =
          con.getClient()
              .execute(
                  query,
                  this,
                  fetchSize,
                  maxRows,
                  resultSetConcurrency,
                  resultSetType,
                  closeOnCompletion,
                  false);
    } catch (SQLException e) {
      results = null;
      currResult = null;
      throw e;
    } finally {
      localInfileInputStream = null;
    }
  }

  /**
   * Executes a prepared batch statement.
   *
   * @return true if the batch was an insert rewrite statement, false otherwise
   */
  private boolean executeInternalPreparedBatch() throws SQLException {
    checkNotClosed();
    Configuration conf = con.getContext().getConf();
    if (conf.rewriteBatchedStatements()
        && parser.isRewriteBatchedApplicable()
        && autoGeneratedKeys != Statement.RETURN_GENERATED_KEYS) {
      executeBatchWithInsertRewrite();
      return true;
    } else {
      boolean possibleLoadLocal = con.getContext().hasClientCapability(LOCAL_FILES);
      if (possibleLoadLocal) {
        String sqlUpper = sql.toUpperCase(Locale.ROOT);
        possibleLoadLocal =
            sqlUpper.contains(" LOCAL ")
                && sqlUpper.contains("LOAD")
                && sqlUpper.contains(" INFILE");
      }
      if (possibleLoadLocal) {
        executeBatchStd();
      } else {
        executeBatchPipeline();
      }
      return false;
    }
  }

  /**
   * Send 1 packet for all insert queries.
   *
   * @throws SQLException if IOException / Command error
   */
  private void executeBatchWithInsertRewrite() throws SQLException {
    try {
      results =
          con.getClient()
              .executePipeline(
                  getClientMessageForRewriteBatchedStatement(),
                  this,
                  0,
                  maxRows,
                  ResultSet.CONCUR_READ_ONLY,
                  ResultSet.TYPE_FORWARD_ONLY,
                  closeOnCompletion,
                  false);
    } catch (SQLException bue) {
      results = null;
      throw bue;
    }
  }

  private ClientMessage[] getClientMessageForRewriteBatchedStatement() {
    RewriteClientParser rewriteClientParser = RewriteClientParser.rewritableParts(parser.getSql());
    return new ClientMessage[] {
      new RewriteQueryMultiPacket(
          con.getContext().getConf(),
          rewriteClientParser.getParamCount(),
          rewriteClientParser,
          batchParameters)
    };
  }

  /**
   * Send n * COM_QUERY + n * read answer
   *
   * @throws SQLException if IOException / Command error
   */
  private void executeBatchPipeline() throws SQLException {
    ClientMessage[] packets = new ClientMessage[batchParameters.size()];
    for (int i = 0; i < batchParameters.size(); i++) {
      packets[i] = new QueryWithParametersPacket(null, parser, batchParameters.get(i), null);
    }
    try {
      results =
          con.getClient()
              .executePipeline(
                  packets,
                  this,
                  0,
                  maxRows,
                  ResultSet.CONCUR_READ_ONLY,
                  ResultSet.TYPE_FORWARD_ONLY,
                  closeOnCompletion,
                  false);
    } catch (SQLException bue) {
      results = null;
      throw bue;
    }
  }

  /**
   * Send n * (COM_QUERY + read answer)
   *
   * @throws SQLException if IOException / Command error
   */
  private void executeBatchStd() throws SQLException {
    int i = 0;
    try {
      results = new ArrayList<>();
      for (; i < batchParameters.size(); i++) {
        results.addAll(
            con.getClient()
                .execute(
                    new QueryWithParametersPacket(
                        null, parser, batchParameters.get(i), localInfileInputStream),
                    this,
                    0,
                    maxRows,
                    ResultSet.CONCUR_READ_ONLY,
                    ResultSet.TYPE_FORWARD_ONLY,
                    closeOnCompletion,
                    false));
      }
    } catch (SQLException bue) {
      BatchUpdateException exception =
          exceptionFactory().createBatchUpdate(results, batchParameters.size(), bue);
      results = null;
      throw exception;
    }
  }

  /**
   * Executes the SQL statement in this <code>PreparedStatement</code> object, which may be any kind
   * of SQL statement. Some prepared statements return multiple results; the <code>execute</code>
   * method handles these complex statements as well as the simpler form of statements handled by
   * the methods <code>executeQuery</code> and <code>executeUpdate</code>.
   *
   * <p>The <code>execute</code> method returns a <code>boolean</code> to indicate the form of the
   * first result. You must call either the method <code>getResultSet</code> or <code>getUpdateCount
   * </code> to retrieve the result; you must call <code>getMoreResults</code> to move to any
   * subsequent result(s).
   *
   * @return <code>true</code> if the first result is a <code>ResultSet</code> object; <code>false
   *     </code> if the first result is an update count or there is no result
   * @throws SQLException if a database access error occurs; this method is called on a closed
   *     <code>PreparedStatement</code> or an argument is supplied to this method
   * @throws SQLTimeoutException when the driver has determined that the timeout value that was
   *     specified by the {@code setQueryTimeout} method has been exceeded and has at least
   *     attempted to cancel the currently running {@code Statement}
   * @see Statement#execute
   * @see Statement#getResultSet
   * @see Statement#getUpdateCount
   * @see Statement#getMoreResults
   */
  @Override
  public boolean execute() throws SQLException {
    executeInternal();
    currResult = results.remove(0);
    return currResult instanceof Result;
  }

  /**
   * Executes the SQL query in this <code>PreparedStatement</code> object and returns the <code>
   * ResultSet</code> object generated by the query.
   *
   * @return a <code>ResultSet</code> object that contains the data produced by the query; never
   *     <code>null</code>
   * @throws SQLException if a database access error occurs; this method is called on a closed
   *     <code>PreparedStatement</code> or the SQL statement does not return a <code>ResultSet
   *     </code> object
   * @throws SQLTimeoutException when the driver has determined that the timeout value that was
   *     specified by the {@code setQueryTimeout} method has been exceeded and has at least
   *     attempted to cancel the currently running {@code Statement}
   */
  @Override
  public ResultSet executeQuery() throws SQLException {
    executeInternal();
    currResult = results.remove(0);
    if (currResult instanceof Result) {
      return (Result) currResult;
    }
    if (Boolean.parseBoolean(
        con.getContext().getConf().nonMappedOptions().getProperty("permitNoResults", "false"))) {
      return new CompleteResult(
          new ColumnDecoder[0], new byte[0][], con.getContext(), resultSetType);
    }

    throw new SQLException(
        "PrepareStatement.executeQuery() command does NOT return a result-set as expected. Either"
            + " use PrepareStatement.execute(), PrepareStatement.executeUpdate(), or correct"
            + " command");
  }

  private ExceptionFactory exceptionFactory() {
    return con.getExceptionFactory().of(this);
  }

  /**
   * Executes the SQL statement in this <code>PreparedStatement</code> object, which must be an SQL
   * Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   * <code>DELETE</code>; or an SQL statement that returns nothing, such as a DDL statement.
   *
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements or (2) 0
   *     for SQL statements that return nothing
   * @throws SQLException if a database access error occurs; this method is called on a closed
   *     <code>PreparedStatement</code> or the SQL statement returns a <code>ResultSet</code> object
   * @throws SQLTimeoutException when the driver has determined that the timeout value that was
   *     specified by the {@code setQueryTimeout} method has been exceeded and has at least
   *     attempted to cancel the currently running {@code Statement}
   */
  @Override
  public int executeUpdate() throws SQLException {
    return (int) executeLargeUpdate();
  }

  /**
   * Executes the SQL statement in this <code>PreparedStatement</code> object, which must be an SQL
   * Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   * <code>DELETE</code>; or an SQL statement that returns nothing, such as a DDL statement.
   *
   * <p>This method should be used when the returned row count may exceed {@link Integer#MAX_VALUE}.
   *
   * <p>The default implementation will throw {@code UnsupportedOperationException}
   *
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements or (2) 0
   *     for SQL statements that return nothing
   * @throws SQLException if a database access error occurs; this method is called on a closed
   *     <code>PreparedStatement</code> or the SQL statement returns a <code>ResultSet</code> object
   * @throws SQLTimeoutException when the driver has determined that the timeout value that was
   *     specified by the {@code setQueryTimeout} method has been exceeded and has at least
   *     attempted to cancel the currently running {@code Statement}
   * @since 1.8
   */
  @Override
  public long executeLargeUpdate() throws SQLException {
    executeInternal();
    currResult = results.remove(0);
    if (currResult instanceof Result) {
      throw exceptionFactory()
          .create("the given SQL statement produces an unexpected ResultSet object", "HY000");
    }
    return ((OkPacket) currResult).getAffectedRows();
  }

  /**
   * Adds a set of parameters to this <code>PreparedStatement</code> object's batch of commands.
   *
   * @throws SQLException if a database access error occurs or this method is called on a closed
   *     <code>PreparedStatement</code>
   * @see Statement#addBatch
   * @since 1.2
   */
  @Override
  public void addBatch() throws SQLException {
    validParameters();
    if (batchParameters == null) batchParameters = new ArrayList<>();
    batchParameters.add(parameters);
    parameters = parameters.clone();
  }

  /**
   * Empties this <code>PreparedStatement</code> object's current list of parameters.
   *
   * @throws SQLException if a database access error occurs, this method is called on a closed
   *     <code>Statement</code> or the driver does not support batch updates
   * @see #addBatch
   * @see DatabaseMetaData#supportsBatchUpdates
   * @since 1.2
   */
  @Override
  public void clearBatch() throws SQLException {
    checkNotClosed();
    if (batchParameters == null) {
      batchParameters = new ArrayList<>();
    } else {
      batchParameters.clear();
    }
  }

  protected void validParameters() throws SQLException {
    for (int i = 0; i < parser.getParamCount(); i++) {
      if (!parameters.containsKey(i)) {
        throw exceptionFactory()
            .create("Parameter at position " + (i + 1) + " is not set", "07004");
      }
    }
  }

  @Override
  public void setQueryTimeout(int seconds) throws SQLException {
    super.setQueryTimeout(seconds);
    if (prepareResult != null) {
      prepareResult.close(con.getClient());
      prepareResult = null;
    }
  }

  @Override
  public void setMaxRows(int max) throws SQLException {
    super.setMaxRows(max);
    if (prepareResult != null) {
      prepareResult.close(con.getClient());
      prepareResult = null;
    }
  }

  @Override
  public void setLargeMaxRows(long max) throws SQLException {
    super.setLargeMaxRows(max);
    if (prepareResult != null) {
      prepareResult.close(con.getClient());
      prepareResult = null;
    }
  }

  /**
   * Retrieves a <code>ResultSetMetaData</code> object that contains information about the columns
   * of the <code>ResultSet</code> object that will be returned when this <code>PreparedStatement
   * </code> object is executed.
   *
   * <p>Because a <code>PreparedStatement</code> object is precompiled, it is possible to know about
   * the <code>ResultSet</code> object that it will return without having to execute it.
   * Consequently, it is possible to invoke the method <code>getMetaData</code> on a <code>
   * PreparedStatement</code> object rather than waiting to execute it and then invoking the <code>
   * ResultSet.getMetaData</code> method on the <code>ResultSet</code> object that is returned.
   *
   * <p><B>NOTE:</B> Using this method may be expensive for some drivers due to the lack of
   * underlying DBMS support.
   *
   * @return the description of a <code>ResultSet</code> object's columns or <code>null</code> if
   *     the driver cannot return a <code>ResultSetMetaData</code> object
   * @throws SQLException if a database access error occurs or this method is called on a closed
   *     <code>PreparedStatement</code>
   * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.2
   */
  @Override
  public ResultSetMetaData getMetaData() throws SQLException {

    // send COM_STMT_PREPARE
    if (prepareResult == null)
      con.getClient().execute(new PreparePacket(escapeTimeout(sql)), this, true);
    return new com.singlestore.jdbc.client.result.ResultSetMetaData(
        exceptionFactory(), prepareResult.getColumns(), con.getContext().getConf(), false);
  }

  /**
   * Retrieves the number, types and properties of this <code>PreparedStatement</code> object's
   * parameters.
   *
   * @return a <code>ParameterMetaData</code> object that contains information about the number,
   *     types and properties for each parameter marker of this <code>PreparedStatement</code>
   *     object
   * @throws SQLException if a database access error occurs or this method is called on a closed
   *     <code>PreparedStatement</code>
   * @see ParameterMetaData
   * @since 1.4
   */
  @Override
  public ParameterMetaData getParameterMetaData() throws SQLException {
    // send COM_STMT_PREPARE
    if (prepareResult == null) {
      con.getClient().execute(new PreparePacket(escapeTimeout(sql)), this, true);
    }
    return new ParameterMetaData(exceptionFactory(), prepareResult.getParameters());
  }

  @Override
  @SuppressWarnings("try")
  public int[] executeBatch() throws SQLException {
    checkNotClosed();
    if (batchParameters == null || batchParameters.isEmpty()) return new int[0];
    try (ClosableLock ignore = lock.closeableLock();
        QueryTimeoutHandler ignore2 = this.con.handleTimeout(queryTimeout)) {
      boolean rewrite = executeInternalPreparedBatch();
      int[] updates = getUpdateCounts(rewrite);
      currResult = results.remove(0);
      return updates;
    } catch (SQLException e) {
      results = null;
      currResult = null;
      throw e;
    } finally {
      batchParameters.clear();
    }
  }

  private int[] getUpdateCounts(boolean rewrite) {
    int[] result;
    if (rewrite) {
      result = new int[batchParameters.size()];
      int resultValue = SUCCESS_NO_INFO;
      if (results.get(0) instanceof OkPacket) {
        int affectedRows = (int) ((OkPacket) results.get(0)).getAffectedRows();
        if (batchParameters.size() == 1 || affectedRows == 0) {
          resultValue = affectedRows;
        }
      }
      Arrays.fill(result, resultValue);
    } else {
      result = new int[results.size()];
      for (int i = 0; i < results.size(); i++) {
        if (results.get(i) instanceof OkPacket) {
          result[i] = (int) ((OkPacket) results.get(i)).getAffectedRows();
        } else {
          result[i] = SUCCESS_NO_INFO;
        }
      }
    }
    return result;
  }

  @Override
  @SuppressWarnings("try")
  public long[] executeLargeBatch() throws SQLException {
    checkNotClosed();
    if (batchParameters == null || batchParameters.isEmpty()) return new long[0];
    try (ClosableLock ignore = lock.closeableLock();
        QueryTimeoutHandler ignore2 = this.con.handleTimeout(queryTimeout)) {
      boolean rewrite = executeInternalPreparedBatch();
      long[] updates = getLargeUpdateCounts(rewrite);
      currResult = results.remove(0);
      return updates;

    } catch (SQLException e) {
      results = null;
      currResult = null;
      throw e;
    } finally {
      batchParameters.clear();
    }
  }

  private long[] getLargeUpdateCounts(boolean rewrite) {
    long[] result;
    if (rewrite) {
      result = new long[batchParameters.size()];
      long resultValue = SUCCESS_NO_INFO;
      if (results.get(0) instanceof OkPacket) {
        long affectedRows = ((OkPacket) results.get(0)).getAffectedRows();
        if (batchParameters.size() == 1 || affectedRows == 0) {
          resultValue = affectedRows;
        }
      }
      Arrays.fill(result, resultValue);
    } else {
      result = new long[results.size()];
      for (int i = 0; i < results.size(); i++) {
        if (results.get(i) instanceof OkPacket) {
          result[i] = ((OkPacket) results.get(i)).getAffectedRows();
        } else {
          result[i] = SUCCESS_NO_INFO;
        }
      }
    }
    return result;
  }

  @Override
  @SuppressWarnings("try")
  public void close() throws SQLException {
    if (prepareResult != null) {
      try (ClosableLock ignore = lock.closeableLock()) {
        prepareResult.close(this.con.getClient());
      }
    }
    con.fireStatementClosed(this);
    super.close();
  }

  @Override
  public String toString() {
    return "ClientPreparedStatement{" + super.toString() + '}';
  }
}
