// SPDX-License-Identifier: LGPL-2.1-or-later
// Copyright (c) 2012-2014 Monty Program Ab
// Copyright (c) 2015-2025 MariaDB Corporation Ab
// Copyright (c) 2021-2025 SingleStore, Inc.

package com.singlestore.jdbc.message.server;

import com.singlestore.jdbc.client.Column;
import com.singlestore.jdbc.client.DataType;
import com.singlestore.jdbc.client.ReadableByteBuf;
import com.singlestore.jdbc.message.ServerMessage;
import com.singlestore.jdbc.util.CharsetEncodingLength;
import com.singlestore.jdbc.util.constants.ColumnFlags;
import java.util.Objects;

/** Column metadata definition */
public class ColumnDefinitionPacket implements Column, ServerMessage {

  private final ReadableByteBuf buf;
  protected final int charset;
  protected final long columnLength;
  protected final DataType dataType;
  protected final byte decimals;
  private final int flags;
  private final int[] stringPos;
  protected final String extTypeName;
  protected final String extTypeFormat;
  /** configuration: use alias as name */
  private boolean useAliasAsName;

  /**
   * Column definition constructor
   *
   * @param buf buffer
   * @param charset charset
   * @param columnLength maxium column length
   * @param dataType data type
   * @param decimals decimal length
   * @param flags flags
   * @param stringPos string position indexes
   * @param extTypeName extended type name
   * @param extTypeFormat extended type format
   * @param useAliasAsName use alias as name
   */
  public ColumnDefinitionPacket(
      ReadableByteBuf buf,
      int charset,
      long columnLength,
      DataType dataType,
      byte decimals,
      int flags,
      int[] stringPos,
      String extTypeName,
      String extTypeFormat,
      boolean useAliasAsName) {
    this.buf = buf;
    this.charset = charset;
    this.columnLength = columnLength;
    this.dataType = dataType;
    this.decimals = decimals;
    this.flags = flags;
    this.stringPos = stringPos;
    this.extTypeName = extTypeName;
    this.extTypeFormat = extTypeFormat;
    this.useAliasAsName = useAliasAsName;
  }

  protected ColumnDefinitionPacket(ColumnDefinitionPacket prev, boolean useAliasAsName) {
    this.buf = prev.buf;
    this.charset = prev.charset;
    this.columnLength = prev.columnLength;
    this.dataType = prev.dataType;
    this.decimals = prev.decimals;
    this.flags = prev.flags;
    this.stringPos = prev.stringPos;
    this.extTypeName = prev.extTypeName;
    this.extTypeFormat = prev.extTypeFormat;
    this.useAliasAsName = useAliasAsName;
  }

  public String getCatalog() {
    return "def";
  }

  @Override
  public String getSchema() {
    buf.pos(stringPos[0]);
    return buf.readString(buf.readIntLengthEncodedNotNull());
  }

  @Override
  public String getTableAlias() {
    buf.pos(stringPos[1]);
    return buf.readString(buf.readIntLengthEncodedNotNull());
  }

  @Override
  public String getTable() {
    buf.pos(stringPos[useAliasAsName ? 1 : 2]);
    return buf.readString(buf.readIntLengthEncodedNotNull());
  }

  @Override
  public String getColumnAlias() {
    buf.pos(stringPos[3]);
    return buf.readString(buf.readIntLengthEncodedNotNull());
  }

  @Override
  public String getColumnName() {
    buf.pos(stringPos[4]);
    return buf.readString(buf.readIntLengthEncodedNotNull());
  }

  @Override
  public long getColumnLength() {
    return columnLength;
  }

  @Override
  public DataType getType() {
    return dataType;
  }

  @Override
  public byte getDecimals() {
    if (dataType == DataType.DATE) {
      return 0;
    }
    return decimals;
  }

  @Override
  public boolean isSigned() {
    return (flags & ColumnFlags.UNSIGNED) == 0;
  }

  @Override
  public int getDisplaySize() {
    switch (dataType) {
      case VARCHAR:
      case ENUM:
      case SET:
      case CHAR:
        Integer maxWidth = CharsetEncodingLength.maxCharlen.get(charset);
        if (maxWidth == null) {
          return (int) Long.min(columnLength, Integer.MAX_VALUE);
        }
        return (int)
            Long.min(Long.divideUnsigned(columnLength, maxWidth.longValue()), Integer.MAX_VALUE);
      case DATE:
        return 10;
      case DATETIME:
      case TIMESTAMP:
        // S2 sends the same length of DATETIME(6) for both DATETIME(0) and DATETIME(6)
        // However in reality DATETIME(0) is 7 symbols shorter as it is missing ".000000"
        return (decimals == 0) ? (int) columnLength - 7 : (int) columnLength;
      case TIME:
        // same as above, but S2 returns 18 instead 17 for some reason
        return (decimals == 0) ? 10 : 17;

      case FLOAT:
        return 12;
      case DOUBLE:
        return 18;
      case MEDIUMINT:
        return Math.min(8, (int) columnLength);

      default:
        return (int) Long.min(columnLength, Integer.MAX_VALUE);
    }
  }

  @Override
  public boolean isPrimaryKey() {
    return (this.flags & ColumnFlags.PRIMARY_KEY) > 0;
  }

  @Override
  public boolean isAutoIncrement() {
    return (this.flags & ColumnFlags.AUTO_INCREMENT) > 0;
  }

  @Override
  public boolean hasDefault() {
    return (this.flags & ColumnFlags.NO_DEFAULT_VALUE_FLAG) == 0;
  }

  // doesn't use & 128 bit filter, because char binary and varchar binary are not binary (handle
  // like string), but have the binary flag
  @Override
  public boolean isBinary() {
    return charset == 63;
  }

  @Override
  public int getFlags() {
    return flags;
  }

  @Override
  public String getExtTypeName() {
    return extTypeName;
  }

  @Override
  public String getExtTypeFormat() {
    return extTypeFormat;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    ColumnDefinitionPacket that = (ColumnDefinitionPacket) o;
    return charset == that.charset
        && columnLength == that.columnLength
        && dataType == that.dataType
        && decimals == that.decimals
        && flags == that.flags;
  }

  @Override
  public int hashCode() {
    return Objects.hash(charset, columnLength, dataType, decimals, flags);
  }
}
