package com.skyflow.sdk.api.service;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;

/**
 * Default interface that allows access to the Skyflow Gateway API.
 */
public interface GatewayService {

    /**
     * Tokenizes a single or bulk group of fields identified in a particular connection by invoking the Skyflow Gateway.
     *
     * @param bearerToken  The authorization token.
     * @param connectionId The ID of the connection to use.
     * @param relativePath The relative path of the connection to use.
     * @param fields       A Map&lt;String, String&gt; of &lt;name, value&gt; field pairs for single tokenization
     *                     or Map&lt;String, List&lt;Map&lt;String, String&gt; of &lt;bulk root field name, list of fields &lt;name, value&gt;&gt; in case of bulk tokenization.
     * @return Map A {@link Map} containing the name of the tokenized fields as key and the token as value in case of single tokenization or list of fields map in case of bulk tokenization.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    Map<String, Object> tokenize(String bearerToken,
                                 String connectionId,
                                 String relativePath,
                                 Map<String, Object> fields) throws IOException, TimeoutException;

    /**
     * Detokenizes a single or bulk group of fields identified in a particular connection by invoking the Skyflow Gateway.
     *
     * @param bearerToken  The authorization token.
     * @param connectionId The ID of the connection to use.
     * @param relativePath The relative path of the connection to use.
     * @param fields       A Map&lt;String, String&gt; of &lt;name, value&gt; field pairs for single detokenization
     *                     or Map&lt;String, List&lt;Map&lt;String, String&gt; of &lt;bulk root field name, list of fields &lt;name, tokenized value&gt;&gt; in case of bulk detokenization.
     * @return Map A {@link Map} containing the name of the tokenized fields as key and detokenized value in case of single detokenization or list of fields map in case of bulk detokenization.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    Map<String, Object> detokenize(String bearerToken,
                                   String connectionId,
                                   String relativePath,
                                   Map<String, Object> fields) throws IOException, TimeoutException;
}
