package com.skyflow.sdk.api.service;

import com.skyflow.sdk.api.model.AuthKey;
import com.skyflow.sdk.api.model.Route;
import com.skyflow.sdk.internal.model.RoutesResponse;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.TimeoutException;

import static java.time.temporal.ChronoUnit.MINUTES;

/**
 * Default interface that allows access to the Management API in Skyflow.
 */
public interface ManagementService {

    /**
     * Simplified version of {@link ManagementService#getBearerToken(Path, Instant)} that uses
     * the default expiration time of 30 minutes from now.
     *
     * @param credentialsFilePath Path to credentials.json file.
     * @return String A bearer token from Skyflow auth service.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    default String getBearerToken(Path credentialsFilePath) throws IOException, TimeoutException {
        return getBearerToken(credentialsFilePath, Instant.now().plus(30, MINUTES));
    }

    /**
     * Retrieves a token from Skyflow authentication service based on credentials.json file and specific expiration time.
     *
     * @param credentialsFilePath Path to credentials.json file.
     * @param expirationTime      JWT token expiration time.
     * @return String A bearer token from Skyflow auth service.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    String getBearerToken(Path credentialsFilePath, Instant expirationTime) throws IOException, TimeoutException;

    /**
     * Simplified version of {@link ManagementService#getBearerToken(String, String, String, Instant)} that uses
     * the default expiration time of 30 minutes from now.
     *
     * @param clientId   ID associated with the service account used.
     * @param apiKey     API key.
     * @param privateKey Private key that will be used to sign the request.
     * @return String A bearer token from Skyflow auth service.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    default String getBearerToken(String clientId, String apiKey, String privateKey) throws IOException, TimeoutException {
        return getBearerToken(clientId, apiKey, privateKey, Instant.now().plus(30, MINUTES));
    }

    /**
     * Retrieves a token from Skyflow authentication service for the given input credentials.
     *
     * @param clientID       ID associated with the service account used.
     * @param apiKey         API key.
     * @param privateKey     Private key that will be used to sign the request.
     * @param expirationTime JWT token expiration time.
     * @return String A bearer token from Skyflow auth service.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    String getBearerToken(String clientID, String apiKey, String privateKey, Instant expirationTime) throws IOException, TimeoutException;

    /**
     * Get the auth public keys.
     *
     * @param bearerToken The token used to authenticate the request.
     * @return List The {@link AuthKey}s.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    List<AuthKey> getAuthKeys(String bearerToken) throws IOException, TimeoutException;

    /**
     * Returns all information about paths in specific connection.
     *
     * @param bearerToken  The authorization token.
     * @param connectionId The ID of the connection to use.
     * @return A {@link RoutesResponse} object.
     * @throws IOException      Thrown if an input/output exception occurred while executing this method.
     * @throws TimeoutException Thrown if a timeout occurs.
     */
    List<Route> getRoutes(String bearerToken, String connectionId) throws IOException, TimeoutException;
}
