package com.skyflow.sdk.internal.http;

import com.skyflow.sdk.api.http.HttpMethod;
import com.skyflow.sdk.api.http.HttpRequest;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Stream;

import static com.skyflow.sdk.api.http.HttpRequest.AUTHORIZATION_HEADER;
import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.IOUtils.toInputStream;

public class HttpRequestBuilder {
    private final HttpMethod method;
    private final String url;
    private final Map<String, String> pathParams = new HashMap<>();
    private final Map<String, List<String>> queryParams = new HashMap<>();
    private final Map<String, List<String>> headers = new HashMap<>();
    private InputStream body;

    public HttpRequestBuilder(HttpMethod method, String url) {
        this.method = method;
        this.url = url;
    }

    public HttpRequestBuilder withPathParam(String key, String value) {
        this.pathParams.put(key, value);
        return this;
    }

    public HttpRequestBuilder withQueryParams(String key, String... values) {
        return appendToMap(this.queryParams, key, values);
    }

    public HttpRequestBuilder withQueryParams(Map<String, List<String>> queryParams) {
        return appendToMap(this.queryParams, queryParams);
    }

    public HttpRequestBuilder withJSonContentType() {
        return withHeaders("Content-Type", "application/json");
    }

    public HttpRequestBuilder withAuthorization(String token) {
        return withHeaders(AUTHORIZATION_HEADER, token);
    }

    public HttpRequestBuilder withAuthorizationBearer(String token) {
        return withHeaders(AUTHORIZATION_HEADER, format("Bearer %s", token));
    }

    public HttpRequestBuilder withHeaders(String key, String... values) {
        return appendToMap(this.headers, key, values);
    }

    public HttpRequestBuilder withHeaders(Map<String, List<String>> headers) {
        return appendToMap(this.headers, headers);
    }

    public HttpRequestBuilder withBody(InputStream body) {
        this.body = body;
        return this;
    }

    public HttpRequestBuilder withBody(String body) {
        this.body = toInputStream(body, defaultCharset());
        return this;
    }

    private HttpRequestBuilder appendToMap(Map<String, List<String>> map, String key, String... values) {
        map.computeIfAbsent(key, element -> new ArrayList<>()).addAll(Stream.of(values).collect(toList()));
        return this;
    }

    private HttpRequestBuilder appendToMap(Map<String, List<String>> map, Map<String, List<String>> newValues) {
        Optional.ofNullable(newValues).ifPresent(map::putAll);
        return this;
    }

    public HttpRequest build() {
        String computedUrl = this.url;
        for (Map.Entry<String, String> pathParam : pathParams.entrySet()) {
            computedUrl = computedUrl.replaceAll(format("\\{%s\\}", pathParam.getKey()), pathParam.getValue());
        }
        return new SimpleHttpRequest(this.method, computedUrl, this.queryParams, this.headers, this.body);
    }

}
