package com.skyflow.sdk.internal.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.skyflow.sdk.api.exception.authentication.InvalidBearerTokenException;
import com.skyflow.sdk.api.exception.http.InvalidConnectionIdException;
import com.skyflow.sdk.api.exception.http.NotFoundException;
import com.skyflow.sdk.api.exception.validation.InvalidBaseUrlException;
import com.skyflow.sdk.api.exception.validation.InvalidConnectionIdValidationException;
import com.skyflow.sdk.api.exception.validation.InvalidFieldsException;
import com.skyflow.sdk.api.exception.validation.InvalidRelativePathException;
import com.skyflow.sdk.api.http.HttpClient;
import com.skyflow.sdk.api.service.GatewayService;
import com.skyflow.sdk.internal.http.HttpRequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeoutException;

import static com.skyflow.sdk.api.http.HttpMethod.POST;
import static com.skyflow.sdk.api.http.HttpRequest.X_SKYFLOW_AUTHORIZATION;
import static com.skyflow.sdk.internal.utils.Validator.validateNotEmpty;
import static com.skyflow.sdk.internal.utils.Validator.validateUrl;
import static java.lang.String.format;

/**
 * Default implementation of {@link GatewayService}.
 */
public class GatewayServiceImpl extends SkyflowService implements GatewayService {
    private static final Logger logger = LoggerFactory.getLogger(GatewayServiceImpl.class);
    private final String baseUrl;

    public GatewayServiceImpl(String baseUrl, HttpClient client, ObjectMapper objectMapper) {
        super(client, objectMapper);
        this.baseUrl = normalizePathPart(baseUrl);
        validateUrl(this.baseUrl, InvalidBaseUrlException::new);
    }

    /**
     * @see GatewayService#tokenize(String, String, String, Map)
     */
    @Override
    public Map<String, Object> tokenize(String bearerToken, String connectionId, String relativePath, Map<String, Object> fields) throws IOException, TimeoutException {
        logger.info("Applying tokenization");
        return sendGatewayRequest(bearerToken, connectionId, relativePath, fields);
    }

    /**
     * @see GatewayService#detokenize(String, String, String, Map)
     */
    @Override
    public Map<String, Object> detokenize(String bearerToken, String connectionId, String relativePath, Map<String, Object> fields) throws IOException, TimeoutException {
        logger.info("Applying detokenization");
        return sendGatewayRequest(bearerToken, connectionId, relativePath, fields);
    }

    private Map<String, Object> sendGatewayRequest(String bearerToken,
                                                   String connectionId,
                                                   String relativePath,
                                                   Map<String, Object> fields) throws IOException, TimeoutException {
        logger.debug("Validating bearer token for emptiness.");
        validateNotEmpty(bearerToken, InvalidBearerTokenException::new);
        logger.debug("Validating connection ID for emptiness.");
        validateNotEmpty(normalizePathPart(connectionId), InvalidConnectionIdValidationException::new);
        logger.debug("Validating relative path for emptiness.");
        validateNotEmpty(relativePath, InvalidRelativePathException::new);
        String url = format("%s/v1/gateway/inboundRoutes/%s/%s", baseUrl, normalizePathPart(connectionId), normalizePathPart(relativePath));
        logger.debug("Validating full URL.");
        validateUrl(url, InvalidRelativePathException::new);
        logger.debug("Validating fields map for emptiness.");
        validateNotEmpty(fields, InvalidFieldsException::new);
        try {
            return unmarshall(send(new HttpRequestBuilder(POST, url)
                    .withHeaders(X_SKYFLOW_AUTHORIZATION, bearerToken.trim())
                    .withJSonContentType()
                    .withBody(marshall(fields))
                    .build()), Map.class);
        } catch (NotFoundException e) {
            Optional.of(e.getMessage())
                    .filter(s -> s.contains("Error Reading Document"))
                    .ifPresent(s -> {
                        throw new InvalidConnectionIdException("Invalid connection id", e);
                    });
            throw e;
        }
    }

    private String normalizePathPart(String path) {
        if (path != null && !path.isEmpty()) {
            path = path.trim();
            if (path.startsWith("/")) {
                path = path.substring(1);
            }
            if (path.endsWith("/")) {
                path = path.substring(0, path.length() - 1);
            }
        }
        return path;
    }
}

