package com.skyflow.sdk.internal.service;

import com.auth0.jwt.JWT;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.skyflow.sdk.api.exception.authentication.InvalidBearerTokenException;
import com.skyflow.sdk.api.exception.authentication.JWTAuthenticationException;
import com.skyflow.sdk.api.exception.parse.ParsingException;
import com.skyflow.sdk.api.exception.validation.*;
import com.skyflow.sdk.api.http.HttpClient;
import com.skyflow.sdk.api.model.AuthKey;
import com.skyflow.sdk.api.model.Route;
import com.skyflow.sdk.api.service.ManagementService;
import com.skyflow.sdk.internal.http.HttpRequestBuilder;
import com.skyflow.sdk.internal.model.GetAuthKeysResponse;
import com.skyflow.sdk.internal.model.JwtCredentials;
import com.skyflow.sdk.internal.model.RoutesResponse;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.nio.file.Path;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeoutException;

import static com.auth0.jwt.algorithms.Algorithm.RSA256;
import static com.skyflow.sdk.api.http.HttpMethod.GET;
import static com.skyflow.sdk.api.http.HttpMethod.POST;
import static com.skyflow.sdk.internal.utils.Validator.*;
import static java.lang.String.format;

/**
 * Default implementation of {@link ManagementService}.
 */
public class ManagementServiceImpl extends SkyflowService implements ManagementService {
    private static final Logger logger = LoggerFactory.getLogger(ManagementServiceImpl.class);
    private static final String TOKEN_URI = "https://manage.skyflowapis.com/v1/auth/sa/oauth/token";

    public ManagementServiceImpl(HttpClient client, ObjectMapper objectMapper) {
        super(client, objectMapper);
    }

    /**
     * @see ManagementService#getBearerToken(Path, Instant)
     */
    @Override
    public String getBearerToken(Path credentialsFilePath, Instant expirationTime) throws IOException, TimeoutException {
        logger.debug("Validating that the credentials file path represents a file that actually exists and can be read.");
        validateExistingFile(credentialsFilePath, InvalidCredentialsFileException::new);
        JwtCredentials jwtCredentials;
        try (InputStream credentialsStream = new FileInputStream(credentialsFilePath.toFile())) {
            jwtCredentials = unmarshall(credentialsStream, JwtCredentials.class);
        } catch (IOException e) {
            throw new ParsingException(e);
        }
        return getBearerToken(jwtCredentials.getClientId(), jwtCredentials.getApiKey(), jwtCredentials.getPrivateKey(), expirationTime);
    }

    /**
     * @see ManagementService#getBearerToken(String, String, String, Instant)
     */
    @Override
    public String getBearerToken(String clientId, String apiKey, String privateKey, Instant expirationTime) throws IOException, TimeoutException {
        try {
            logger.debug("Validating client ID for emptiness.");
            validateNotEmpty(clientId, InvalidClientIdException::new);
            logger.debug("Validating API key for emptiness.");
            validateNotEmpty(apiKey, InvalidApiKeyException::new);
            logger.debug("Validating private key for emptiness.");
            validateNotEmpty(privateKey, InvalidPrivateKeyException::new);
            logger.debug("Validating that expiration time is in the future.");
            validateFutureDate(expirationTime, InvalidExpirationTimeException::new);
            logger.info("Retrieving bearer token.");
            return Optional.ofNullable(send(new HttpRequestBuilder(POST, TOKEN_URI)
                    .withAuthorization(apiKey.trim())
                    .withJSonContentType()
                    .withBody(format("{\"grant_type\": \"urn:ietf:params:oauth:grant-type:jwt-bearer\", \"assertion\": \"%s\"}", JWT.create()
                            .withClaim("iss", clientId.trim())
                            .withClaim("key", apiKey.trim())
                            .withClaim("aud", TOKEN_URI)
                            .withClaim("exp", expirationTime.getEpochSecond())
                            .withClaim("sub", clientId.trim())
                            .sign(RSA256(null, RSAPrivateKey.class.cast(KeyFactory.getInstance("RSA").generatePrivate(new PKCS8EncodedKeySpec(
                                    Optional.of(new PemReader(new StringReader(privateKey.trim())))
                                            .map(pemReader -> {
                                                try {
                                                    return pemReader.readPemObject();
                                                } catch (IOException e) {
                                                    throw new RuntimeException(e);
                                                }
                                            })
                                            .map(PemObject::getContent)
                                            .orElseThrow(InvalidPrivateKeyException::new))))))))
                    .build()))
                    .map(this::unmarshall)
                    .map(tree -> tree.get("accessToken"))
                    .map(JsonNode::textValue)
                    .orElseThrow(ParsingException::new);
        } catch (InvalidBearerTokenException e) {
            switch (e.getMessage().toLowerCase()) {
                case "error: invalid key":
                case "error: crypto/rsa: verification error":
                    throw new InvalidJWTException(e);
                case "error: invalid_grant: invalid exp":
                    throw new InvalidExpirationTimeException(e);
                default:
                    throw e;
            }
        } catch (InvalidKeySpecException e) {
            // FIXME: This is too generic. We need specific exceptions here.
            throw new JWTAuthenticationException("Invalid private key specification.", e);
        } catch (NoSuchAlgorithmException e) {
            // FIXME: This is too generic. We need specific exceptions here.
            throw new JWTAuthenticationException("No provider found for RSA algorithm.", e);
        }
    }

    /**
     * @see ManagementService#getAuthKeys(String)
     */
    @Override
    public List<AuthKey> getAuthKeys(String bearerToken) throws IOException, TimeoutException {
        logger.debug("Validating bearer token for emptiness.");
        validateNotEmpty(bearerToken, InvalidBearerTokenException::new);
        logger.info("Retrieving auth keys.");
        return unmarshall(send(new HttpRequestBuilder(GET, "https://manage.skyflowapis.com/v1/auth/sa/oauth/keys")
                        .withAuthorizationBearer(bearerToken.trim())
                        .build()),
                GetAuthKeysResponse.class)
                .getKeys();
    }

    /**
     * @see ManagementService#getRoutes(String, String) (String, String, String, Map)
     */
    @Override
    public List<Route> getRoutes(String bearerToken, String connectionId) throws IOException, TimeoutException {
        logger.debug("Validating bearer token for emptiness.");
        validateNotEmpty(bearerToken, InvalidBearerTokenException::new);
        logger.debug("Validating connection ID for emptiness.");
        String normalizedConectionId = connectionId;
        if (normalizedConectionId != null && !normalizedConectionId.isEmpty()) {
            normalizedConectionId = normalizedConectionId.trim();
            if (normalizedConectionId.startsWith("/")) {
                normalizedConectionId = normalizedConectionId.substring(1);
            }
        }
        validateNotEmpty(normalizedConectionId, InvalidConnectionIdValidationException::new);
        logger.info("Retrieving routes.");
        return unmarshall(send(new HttpRequestBuilder(GET, "https://manage.skyflowapis.com/v1/gateway/outboundRoutes/{connectionId}")
                .withAuthorizationBearer(bearerToken.trim())
                .withJSonContentType()
                .withPathParam("connectionId", normalizedConectionId)
                .build()), RoutesResponse.class)
                .getRoutes();
    }
}
