package com.skyflow.sdk.internal.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.skyflow.sdk.api.exception.SkyflowException;
import com.skyflow.sdk.api.exception.authentication.InvalidBearerTokenException;
import com.skyflow.sdk.api.exception.http.BadRequestException;
import com.skyflow.sdk.api.exception.http.InternalServerErrorException;
import com.skyflow.sdk.api.exception.http.NotFoundException;
import com.skyflow.sdk.api.exception.parse.ParsingException;
import com.skyflow.sdk.api.http.HttpClient;
import com.skyflow.sdk.api.http.HttpRequest;
import com.skyflow.sdk.api.http.HttpResponse;
import com.skyflow.sdk.internal.model.SkyflowError;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;
import java.util.concurrent.TimeoutException;
import java.util.function.BiFunction;
import java.util.function.Predicate;

import static com.skyflow.sdk.api.http.HttpRequest.AUTHORIZATION_HEADER;
import static com.skyflow.sdk.api.http.HttpRequest.X_SKYFLOW_AUTHORIZATION;
import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.nio.charset.StandardCharsets.UTF_8;

public class SkyflowService {
    private static final Logger logger = LoggerFactory.getLogger(SkyflowService.class);
    private final HttpClient client;
    private final ObjectMapper objectMapper;

    public SkyflowService(HttpClient client, ObjectMapper objectMapper) {
        this.client = client;
        this.objectMapper = objectMapper;
    }

    protected InputStream send(HttpRequest request) throws IOException, TimeoutException {
        HttpResponse response = client.send(request);
        if (response.getStatusCode() != 200) {
            try {
                BiFunction<Integer, String, SkyflowException> exceptionHandler = null;
                switch (response.getStatusCode()) {
                    case 500:
                        exceptionHandler = InternalServerErrorException::new;
                        break;
                    case 400:
                        exceptionHandler = BadRequestException::new;
                        break;
                    case 401:
                        exceptionHandler = (errorCode, errorString) -> new InvalidBearerTokenException(request.getHeaders().get(AUTHORIZATION_HEADER) != null ?
                                request.getHeaders().get(AUTHORIZATION_HEADER).stream().findFirst().map(token -> token.replaceAll("Bearer ", "")).orElse(null) :
                                request.getHeaders().get(X_SKYFLOW_AUTHORIZATION).stream().findFirst().orElse(null),
                                errorString);
                        break;
                    case 404:
                        exceptionHandler = NotFoundException::new;
                        break;
                    default:
                        exceptionHandler = (errorCode, errorString) -> new SkyflowException(format("Error executing request: invalid response status %s.", errorString));
                }
                logger.error("Response from Skyflow API returned {} error code. Parsing and throwing exception.", response.getStatusCode());
                String errorString = IOUtils.toString(response.getBody(), defaultCharset()).trim();
                Integer grpcErrorCode = null;
                if (errorString.startsWith("{")) {
                    SkyflowError error = unmarshall(new ByteArrayInputStream(errorString.getBytes(UTF_8)), SkyflowError.class);
                    errorString = error.getMessage();
                    grpcErrorCode = error.getGrpcCode();
                }
                throw exceptionHandler.apply(grpcErrorCode, errorString);
            } catch (IOException e) {
                throw new ParsingException(e);
            }
        }
        return response.getBody();
    }

    protected <T> T unmarshall(InputStream stream, Class<T> type) {
        try {
            return objectMapper.readValue(Optional.ofNullable(stream)
                    .map(inputStream -> {
                        try {
                            return IOUtils.toString(inputStream, defaultCharset());
                        } catch (IOException e) {
                            throw new ParsingException(e);
                        }
                    })
                    .map(String::trim)
                    .filter(Predicate.isEqual("").negate())
                    .orElse("{}"), type);
        } catch (IOException e) {
            throw new ParsingException(e);
        }
    }

    protected JsonNode unmarshall(InputStream stream) {
        try {
            return objectMapper.readTree(stream);
        } catch (IOException e) {
            throw new ParsingException(e);
        }
    }

    protected InputStream marshall(Object object) {
        try {
            return new ByteArrayInputStream(objectMapper.writeValueAsBytes(object));
        } catch (JsonProcessingException e) {
            throw new ParsingException(e);
        }
    }
}
