package com.skyflow.sdk.internal.utils;

import com.skyflow.sdk.api.exception.SkyflowException;
import com.skyflow.sdk.api.exception.validation.ValidationException;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * Class used entirely to provide common validation functionalities.
 */
public final class Validator {

    /**
     * Default constructor. Doesn't allow this class to be instantiated.
     */
    private Validator() {
        throw new UnsupportedOperationException();
    }

    /**
     * Validates that a String is not empty. If empty, throws the exception supplied by the exceptionSupplier.
     *
     * @param value             The string to check for emptiness.
     * @param exceptionSupplier The exception supplier.
     */
    public static void validateNotEmpty(String value, Supplier<? extends SkyflowException> exceptionSupplier) {
        Optional.ofNullable(value)
                .map(String::trim)
                .filter(Predicate.isEqual("").negate())
                .orElseThrow(exceptionSupplier);
    }

    /**
     * Validates that a Map is not empty. If empty, throws the exception supplied by the exceptionSupplier.
     *
     * @param value             The Map to check for emptiness.
     * @param exceptionSupplier The exception supplier.
     */
    public static void validateNotEmpty(Map value, Supplier<? extends SkyflowException> exceptionSupplier) {
        Optional.ofNullable(value)
                .filter(v -> !v.isEmpty())
                .orElseThrow(exceptionSupplier);
    }

    /**
     * Validates that an Instant is in a point in the future. If null or in the future, throws the exception supplied by
     * the exceptionSupplier.
     *
     * @param value             The instant to check.
     * @param exceptionSupplier The exception supplier.
     */
    public static void validateFutureDate(Instant value, Supplier<? extends ValidationException> exceptionSupplier) {
        Optional.ofNullable(value).filter(exp -> exp.isAfter(Instant.now())).orElseThrow(exceptionSupplier);
    }

    /**
     * Validates that the path determined to is a file that actually exists and can be read. If not, throws the exception supplier by the exceptionSupplier.
     *
     * @param path              The {@link Path} to check.
     * @param exceptionSupplier The exception supplier.
     */
    public static void validateExistingFile(Path path, Supplier<? extends SkyflowException> exceptionSupplier) {
        Optional.ofNullable(path)
                .map(Path::toFile)
                .filter(File::exists)
                .filter(File::isFile)
                .filter(File::canRead)
                .orElseThrow(exceptionSupplier);
    }

    /**
     * Validates that the url is of a valid absolute url format. If not, throws the exception supplier by the exceptionSupplier.
     *
     * @param url               The url to check.
     * @param exceptionSupplier The exception supplier.
     */
    public static void validateUrl(String url, Supplier<? extends SkyflowException> exceptionSupplier) {
        Optional.ofNullable(url)
                .map(String::trim)
                .filter(Predicate.isEqual("").negate())
                .map(u -> {
                    try {
                        return new URI(u);
                    } catch (URISyntaxException e) {
                        throw exceptionSupplier.get();
                    }
                })
                .filter(URI::isAbsolute)
                .orElseThrow(exceptionSupplier);
    }

}
