package com.skyflow.sdk.internal.http;

import com.skyflow.sdk.api.http.HttpMethod;
import com.skyflow.sdk.api.http.HttpRequest;

import java.io.InputStream;
import java.util.*;
import java.util.stream.Stream;

import static com.skyflow.sdk.api.http.HttpRequest.AUTHORIZATION_HEADER;
import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.IOUtils.toInputStream;

/**
 * <p>HttpRequestBuilder class.</p>
 *
 * @author sriv
 */
public class HttpRequestBuilder {
    private final HttpMethod method;
    private final String url;
    private final Map<String, String> pathParams = new HashMap<>();
    private final Map<String, List<String>> queryParams = new HashMap<>();
    private final Map<String, List<String>> headers = new HashMap<>();
    private InputStream body;

    /**
     * <p>Constructor for HttpRequestBuilder.</p>
     *
     * @param method a {@link com.skyflow.sdk.api.http.HttpMethod} object
     * @param url a {@link java.lang.String} object
     */
    public HttpRequestBuilder(HttpMethod method, String url) {
        this.method = method;
        this.url = url;
    }

    /**
     * <p>withPathParam.</p>
     *
     * @param key a {@link java.lang.String} object
     * @param value a {@link java.lang.String} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withPathParam(String key, String value) {
        this.pathParams.put(key, value);
        return this;
    }

    /**
     * <p>withQueryParams.</p>
     *
     * @param key a {@link java.lang.String} object
     * @param values a {@link java.lang.String} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withQueryParams(String key, String... values) {
        return appendToMap(this.queryParams, key, values);
    }

    /**
     * <p>withQueryParams.</p>
     *
     * @param queryParams a {@link java.util.Map} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withQueryParams(Map<String, List<String>> queryParams) {
        return appendToMap(this.queryParams, queryParams);
    }

    /**
     * <p>withJSonContentType.</p>
     *
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withJSonContentType() {
        return withHeaders("Content-Type", "application/json");
    }

    /**
     * <p>withAuthorization.</p>
     *
     * @param token a {@link java.lang.String} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withAuthorization(String token) {
        return withHeaders(AUTHORIZATION_HEADER, token);
    }

    /**
     * <p>withAuthorizationBearer.</p>
     *
     * @param token a {@link java.lang.String} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withAuthorizationBearer(String token) {
        return withHeaders(AUTHORIZATION_HEADER, format("Bearer %s", token));
    }

    /**
     * <p>withHeaders.</p>
     *
     * @param key a {@link java.lang.String} object
     * @param values a {@link java.lang.String} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withHeaders(String key, String... values) {
        return appendToMap(this.headers, key, values);
    }

    /**
     * <p>withHeaders.</p>
     *
     * @param headers a {@link java.util.Map} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withHeaders(Map<String, List<String>> headers) {
        return appendToMap(this.headers, headers);
    }

    /**
     * <p>withBody.</p>
     *
     * @param body a {@link java.io.InputStream} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withBody(InputStream body) {
        this.body = body;
        return this;
    }

    /**
     * <p>withBody.</p>
     *
     * @param body a {@link java.lang.String} object
     * @return a {@link com.skyflow.sdk.internal.http.HttpRequestBuilder} object
     */
    public HttpRequestBuilder withBody(String body) {
        this.body = toInputStream(body, defaultCharset());
        return this;
    }

    private HttpRequestBuilder appendToMap(Map<String, List<String>> map, String key, String... values) {
        map.computeIfAbsent(key, element -> new ArrayList<>()).addAll(Stream.of(values).collect(toList()));
        return this;
    }

    private HttpRequestBuilder appendToMap(Map<String, List<String>> map, Map<String, List<String>> newValues) {
        Optional.ofNullable(newValues).ifPresent(map::putAll);
        return this;
    }

    /**
     * <p>build.</p>
     *
     * @return a {@link com.skyflow.sdk.api.http.HttpRequest} object
     */
    public HttpRequest build() {
        String computedUrl = this.url;
        for (Map.Entry<String, String> pathParam : pathParams.entrySet()) {
            computedUrl = computedUrl.replaceAll(format("\\{%s\\}", pathParam.getKey()), pathParam.getValue());
        }
        return new SimpleHttpRequest(this.method, computedUrl, this.queryParams, this.headers, this.body);
    }

}
