package com.skyflow.sdk.api.service;

import com.skyflow.sdk.api.model.AuthKey;
import com.skyflow.sdk.api.model.Route;
import com.skyflow.sdk.internal.model.RoutesResponse;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.TimeoutException;

import static java.time.temporal.ChronoUnit.MINUTES;

/**
 * Default interface that allows access to the Management API in Skyflow.
 *
 * @author sriv
 */
public interface ManagementService {

    /**
     * Simplified version of {@link com.skyflow.sdk.api.service.ManagementService#getBearerToken(Path, Instant)} that uses
     * the default expiration time of 30 minutes from now.
     *
     * @param credentialsFilePath Path to credentials.json file.
     * @return a bearer token from Skyflow auth service.
     * @throws java.io.IOException if any.
     * @throws java.util.concurrent.TimeoutException if any.
     */
    default String getBearerToken(Path credentialsFilePath) throws IOException, TimeoutException {
        return getBearerToken(credentialsFilePath, Instant.now().plus(30, MINUTES));
    }

    /**
     * Retrieves a token from Skyflow authentication service based on credentials.json file and specific expiration time.
     *
     * @param credentialsFilePath Path to credentials.json file.
     * @param expirationTime      JWT token expiration time.
     * @return a bearer token from Skyflow auth service.
     * @throws java.io.IOException if any.
     * @throws java.util.concurrent.TimeoutException if any.
     */
    String getBearerToken(Path credentialsFilePath, Instant expirationTime) throws IOException, TimeoutException;

    /**
     * Simplified version of {@link com.skyflow.sdk.api.service.ManagementService#getBearerToken(String, String, String, Instant, String...)} that uses
     * the default expiration time of 30 minutes from now.
     *
     * @param clientId   ID associated with the service account used.
     * @param apiKey     API key.
     * @param privateKey Private key that will be used to sign the request.
     * @return a bearer token from Skyflow auth service.
     * @throws java.io.IOException if any.
     * @throws java.util.concurrent.TimeoutException if any.
     */
    default String getBearerToken(String clientId, String apiKey, String privateKey) throws IOException, TimeoutException {
        return getBearerToken(clientId, apiKey, privateKey, Instant.now().plus(30, MINUTES));
    }

    /**
     * Retrieves a token from Skyflow authentication service for the given input credentials.
     *
     * @param clientID       ID associated with the service account used.
     * @param apiKey         API key.
     * @param privateKey     Private key that will be used to sign the request.
     * @param expirationTime JWT token expiration time.
     * @return a bearer token from Skyflow auth service.
     * @param tokenURI a {@link java.lang.String} object
     * @throws java.io.IOException if any.
     * @throws java.util.concurrent.TimeoutException if any.
     */
    String getBearerToken(String clientID, String apiKey, String privateKey, Instant expirationTime, String... tokenURI) throws IOException, TimeoutException;

    /**
     * Get the auth public keys.
     *
     * @param bearerToken The token used to authenticate the request.
     * @return List The {@link com.skyflow.sdk.api.model.AuthKey}s.
     * @throws java.io.IOException if any.
     * @throws java.util.concurrent.TimeoutException if any.
     */
    List<AuthKey> getAuthKeys(String bearerToken) throws IOException, TimeoutException;

    /**
     * Returns all information about paths in specific connection.
     *
     * @param bearerToken  The authorization token.
     * @param connectionId The ID of the connection to use.
     * @return A {@link com.skyflow.sdk.internal.model.RoutesResponse} object.
     * @throws java.io.IOException if any.
     * @throws java.util.concurrent.TimeoutException if any.
     */
    List<Route> getRoutes(String bearerToken, String connectionId) throws IOException, TimeoutException;
}
