/*
 * Copyright (C) 2024. Tony Robalik.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package slack.gradle.artifacts

import java.io.Serializable
import org.gradle.api.NamedDomainObjectProvider
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.attributes.Attribute
import org.gradle.api.file.Directory
import org.gradle.api.file.RegularFile
import org.gradle.api.provider.Provider

/**
 * Used for publishing custom artifacts from a subproject to an aggregating project (often the
 * "root" project). Only for inter-project publishing (e.g., _not_ for publishing to Artifactory).
 * See also [Resolver].
 *
 * Represents a set of tightly coupled [Configuration]s:
 * * A "dependency scope" configuration ([Resolver.declarable]).
 * * A "resolvable" configuration ([Resolver.internal]).
 * * A "consumable" configuration ([external]).
 *
 * Dependencies are _declared_ on [Resolver.declarable] in the aggregating project. Custom artifacts
 * (e.g., not jars), generated by tasks, are published via [publish], which should be used on
 * dependency (artifact-producing) projects.
 *
 * Gradle uses [attributes][ShareableArtifact.attribute] to wire the consumer project's
 * [Resolver.internal] (resolvable) configuration to the producer project's [external] (consumable)
 * configuration, which is itself configured via [publish].
 *
 * @see <a
 *   href="https://docs.gradle.org/current/userguide/cross_project_publications.html#sec:variant-aware-sharing">Variant-aware
 *   sharing of artifacts between projects</a>
 * @see <a
 *   href="https://dev.to/autonomousapps/configuration-roles-and-the-blogging-industrial-complex-21mn">Gradle
 *   configuration roles</a>
 * @see <a
 *   href="https://github.com/autonomousapps/dependency-analysis-gradle-plugin/blob/08c8765157925bbcdfd8f63d8d37fe041561ddb4/src/main/kotlin/com/autonomousapps/internal/artifacts/Publisher.kt">Publisher.kt</a>
 */
internal class Publisher<T : Serializable>(
  project: Project,
  attr: Attribute<T>,
  artifact: T,
  declarableName: String,
  category: String,
) {

  companion object {
    /**
     * Convenience function for creating a [Publisher] for inter-project publishing of
     * [SgpArtifact].
     */
    fun interProjectPublisher(project: Project, sgpArtifact: SgpArtifact) =
      interProjectPublisher(
        project,
        sgpArtifact.attribute,
        sgpArtifact,
        sgpArtifact.declarableName,
        sgpArtifact.category,
      )

    fun <T : ShareableArtifact<T>> interProjectPublisher(
      project: Project,
      attr: Attribute<T>,
      artifact: T,
      declarableName: String,
      category: String,
    ): Publisher<T> {
      project.logger.debug("Creating publisher for $artifact")
      return Publisher(
        project,
        attr,
        artifact,
        declarableName,
        category,
      )
    }
  }

  // Following the naming pattern established by the Java Library plugin.
  // See
  // https://docs.gradle.org/current/userguide/java_library_plugin.html#sec:java_library_configurations_graph
  private val externalName = "${declarableName}Elements"

  /**
   * The plugin will expose dependencies on this configuration, which extends from the declared
   * dependencies.
   */
  private val external: NamedDomainObjectProvider<out Configuration> = run {
    if (project.configurations.findByName(externalName) != null) {
      project.configurations.named(externalName)
    } else {
      project.configurations.consumable(externalName) {
        // This attribute is identical to what is set on the internal/resolvable configuration
        attributes {
          attribute(attr, artifact)
          addCommonAttributes(project, category)
        }
      }
    }
  }

  /**
   * Teach Gradle which thing produces the artifact associated with the external/consumable
   * configuration.
   */
  fun publish(output: Provider<RegularFile>) {
    external.configure { outgoing.artifact(output) }
  }

  /**
   * Teach Gradle which thing produces the artifact associated with the external/consumable
   * configuration.
   */
  fun publishDirs(output: Provider<Directory>) {
    external.configure { outgoing.artifact(output) }
  }
}
