/*
 * Copyright 2025 Solace Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.solace.serdes;

import java.io.Closeable;
import java.io.IOException;
import java.util.Map;

/**
 * Interface to convert serialized payload bytes into pojo objects.
 *
 * @param <T> Generic type argument for the output object on {@link Deserializer#deserialize(String, byte[], Map)} or {@link Deserializer#deserialize(String, byte[])}
 *
 * @since 1.0
 */
public interface Deserializer<T> extends Closeable {

    /**
     * Configures this {@link Deserializer}
     *
     * @param properties the configuration properties
     * @since 1.0
     */
    default void configure(Map<String,?> properties) {
        // default implementation is blank
    }

    /**
     * Closes this {@link Deserializer}
     *
     * @since 1.0
     */
    @Override
    default void close() throws IOException {
        // default implementation is blank
    }

    /**
     * Deserialize payload bytes into business object of type T
     *
     * @param destinationName The topic or queue name to route the object.
     * @param payload         The byte[] of payload from carrier to deserialize into an object of type T.
     * @return constructed object of type T from byte[] of payload
     * @since 1.0
     */
    T deserialize(String destinationName, byte[] payload);

    /**
     * Deserialize payload bytes and headers into business object of type T
     *
     * @param destinationName The topic or queue name to route the object.
     * @param payload         The byte[] of payload from carrier to deserialize into an object of type T.
     * @param headers         Defines additional headers to not include in the payload bytes,
     *                        this can be an in/out parameter and can be modified on call to deserialize if present.
     * @return constructed object of type T from byte[] of payload and headers
     * @since 1.0
     */
    default T deserialize(String destinationName, byte[] payload, Map<String,Object> headers) {
        return deserialize(destinationName, payload);
    }
}