/*
 * Copyright 2025 Solace Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.solace.serdes;

import java.io.Closeable;
import java.io.IOException;
import java.util.Map;

/**
 * Interface to convert pojo objects into serialized payload bytes.
 *
 * @param <T> Generic type argument for the input object on {@link Serializer#serialize(String, Object, Map)} or {@link Serializer#serialize(String, Object)}
 *
 * @since 1.0
 */
public interface Serializer<T> extends Closeable {

    /**
     * Configures this {@link Serializer}
     *
     * @param properties the configuration properties
     * @since 1.0
     */
    default void configure(Map<String,?> properties) {
        // default implementation is blank
    }

    /**
     * Closes this {@link Serializer}
     *
     * @since 1.0
     */
    @Override
    default void close() throws IOException {
        // default implementation is blank
    }

    /**
     * Serializes given object and destination.
     * This is equivalent to invoking {@link Serializer#serialize(String, Object, Map)} with null headers.
     *
     * @param destinationName The topic or queue name to route the object.
     * @param object          Business object to serialize.
     * @return byte[] from object that will be added as carrier payload
     * @since 1.0
     */
    byte[] serialize(String destinationName, T object);

    /**
     * Serializes given object, destination and headers.
     *
     * @param destinationName The topic or queue name to route the object.
     * @param object          Business object to serialize.
     * @param headers         Defines additional header to not include in the returned bytes,
     *                        this can be an in/out parameter and can be modified on call to serialize if present.
     * @return byte[] from object that will be added as carrier payload
     * @since 1.0
     */
    default byte[] serialize(String destinationName, T object, Map<String,Object> headers){
        return serialize(destinationName, object);
    }
}