/*
 * Copyright 2025 Solace Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.solace.serdes;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * A deserializer for converting byte[] to strings using a specified character encoding.
 */
public class StringDeserializer implements Deserializer<String> {
    private Charset encoding = StandardCharsets.UTF_8;
    public static final String SOLACE_DESERIALIZER_STRING_ENCODING = "solace.deserializer.string.encoding";

    /**
     * Configures this {@link StringDeserializer} with the given properties.
     * <p>
     * <STRONG>NOTE: The default encoding is UTF-8.</STRONG>
     *
     * @param properties a map of configuration properties
     * @throws IllegalArgumentException if the charset encoding type is unsupported or the specified charset
     *                                          is unsupported, or when properties is null.
     */
    @Override
    public void configure(Map<String,?> properties) {
        if (properties == null) {
            throw new IllegalArgumentException("properties must not be null");
        }
        Object encodingValue = properties.get(SOLACE_DESERIALIZER_STRING_ENCODING);

        if (encodingValue == null) {
            return;
        }
        encoding = EncodingUtil.getEncoding(encodingValue);
    }

    /**
     * Deserializes the given byte[] to string using the configured character encoding.
     *
     * <p>
     * <STRONG>NOTE: The default encoding is UTF-8.</STRONG>
     *
     * @param destinationName the name of the destination (not used in this implementation)
     * @param payload the byte[] to deserialize
     * @return the deserialized String, or null if the input payload is null
     */
    @Override
    public String deserialize(String destinationName, byte[] payload) {
        if (payload == null) {
            return null;
        }
        return new String(payload, encoding);
    }
}
