/*
 * Copyright 2025 Solace Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.solace.serdes.version;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.CodeSource;
import java.security.ProtectionDomain;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

class VersionInfoImpl implements VersionInfo {

    private static final String UNKNOWN = "--"; // Default for unknown values

    // Use standard Manifest attribute names
    private static final Attributes.Name MF_IMPLEMENTATION_VERSION = Attributes.Name.IMPLEMENTATION_VERSION;
    private static final String MF_BUNDLE_SYMBOLIC_NAME = "Bundle-SymbolicName";
    // Custom header name for SCM Revision (must match maven-bundle-plugin config)
    private static final String MF_SCM_REVISION = "SCM-Revision";

    private final String version;
    private final String revision;
    private final String module;

    /**
     * Creates a new instance of VersionInfoImpl.
     * Reads version and revision information from the JAR manifest.
     */
    VersionInfoImpl() {
        String[] versionInfo = lookupVersionInfo();
        this.version = versionInfo[0];
        this.revision = versionInfo[1];
        this.module = versionInfo[2];
    }

    @Override
    public String getVersion() {
        return this.version;
    }

    @Override
    public String getRevision() {
        return this.revision;
    }

    @Override
    public String getModule() {
        return this.module;
    }

    static String[] lookupVersionInfo() {
        String foundVersion = UNKNOWN;
        String foundRevision = UNKNOWN;
        String foundModule = UNKNOWN;

        try {
            Class<?> clazz = VersionInfoImpl.class;
            ProtectionDomain protectionDomain = clazz.getProtectionDomain();
            if (protectionDomain == null) {
                return new String[]{foundVersion, foundRevision, foundModule};
            }

            CodeSource cs = protectionDomain.getCodeSource();
            if (cs == null) {
                return new String[]{foundVersion, foundRevision, foundModule};
            }

            URL locationUrl = cs.getLocation();
            if (locationUrl == null) {
                return new String[]{foundVersion, foundRevision, foundModule};
            }

            Manifest manifest = readManifest(locationUrl);
            if (manifest == null) {
                return new String[]{foundVersion, foundRevision, foundModule};
            }

            Attributes mainAttributes = manifest.getMainAttributes();
            if (mainAttributes == null) {
                return new String[]{foundVersion, foundRevision, foundModule};
            }

            // Read standard Implementation-Version
            String implVersion = mainAttributes.getValue(MF_IMPLEMENTATION_VERSION);
            // Read custom SCM-Revision header
            String scmRevision = mainAttributes.getValue(MF_SCM_REVISION);
            // Read standard Bundle-SymbolicName
            String bundleSymbolicName = mainAttributes.getValue(MF_BUNDLE_SYMBOLIC_NAME);

            foundVersion = (implVersion != null) ? implVersion : UNKNOWN;
            foundRevision = (scmRevision != null) ? scmRevision : UNKNOWN;
            foundModule = (bundleSymbolicName != null) ? bundleSymbolicName : UNKNOWN;
        } catch (Exception ignored) {
            // Silently handle exceptions
        }

        return new String[]{foundVersion, foundRevision, foundModule};
    }

    /**
     * Reads the manifest from the given location URL.
     * Handles both JAR and directory locations.
     *
     * @param locationUrl The URL of the class location
     * @return The manifest, or null if not found or an error occurred
     */
    static Manifest readManifest(URL locationUrl) {
        try {
            String urlString = locationUrl.toString();
            URL manifestUrl = null;

            if (urlString.endsWith("/")) {
                // Running from a directory (e.g., classes directory)
                manifestUrl = new URL(locationUrl, "META-INF/MANIFEST.MF");
            } else if (urlString.endsWith(".jar")) {
                // Running from a JAR file
                manifestUrl = new URL("jar:" + urlString + "!/META-INF/MANIFEST.MF");
            }

            if (manifestUrl == null) {
                return null;
            }

            try (InputStream manifestStream = manifestUrl.openStream()) {
                return new Manifest(manifestStream);
            }
        } catch (IOException ioe) {
            // Silently handle exceptions
            return null;
        }
    }

    @Override
    public String toString() {
        return "VersionInfo[version=" + version + ", revision=" + revision + ", module=" + module + "]";
    }
}
