@file:Suppress("PackageDirectoryMismatch")

package korlibs.datastructure.algo

import korlibs.concurrent.lock.*
import korlibs.datastructure.*

/**
 * Supports getting a map determining the number of occurrences for different [Int]
 *
 * For example:
 *
 * ```kotlin
 * intArrayOf(1, 1, 5, 1, 9, 5) // 3x 1, 2x 5, 1x 9
 * ==
 * intIntMapOf((1 to 3), (5 to 2), (9 to 1))
 * ```
 **/
class Historiogram(private val out: IntIntMap = IntIntMap()) {
    private val lock = NonRecursiveLock()

    /** Adds a new [value] to this historiogram */
    fun add(value: Int) {
        lock {
            out.getOrPut(value) { 0 }
            out[value]++
        }
    }

    /** Adds a set of [values] in the optional range [start]..<[end] to this historiogram */
    fun addArray(values: IntArray, start: Int = 0, end: Int = values.size) {
        lock {
            for (n in start until end) {
                val value = values[n]
                out.getOrPut(value) { 0 }
                out[value]++
            }
        }

    }

    /** Gets a copy of the map representing each value with its frequency. */
    fun getMapCopy(): IntIntMap {
        val map = IntIntMap()
        lock { out.fastForEach { key, value -> map[key] = value } }
        return map
    }

    /** Creates a new Historiogram */
    fun clone(): Historiogram = Historiogram(getMapCopy())

    override fun toString(): String = "Historiogram(${getMapCopy().toMap()})"

    companion object {
        /**
         * From an IntArray of [values] in the optional range [start], [end].
         * Computes the Historiogram, and returns an [IntIntMap] mapping each different to the number of occurrences.
         **/
        fun values(values: IntArray, start: Int = 0, end: Int = values.size): IntIntMap {
            return Historiogram().also { it.addArray(values, start, end) }.out
        }
    }
}

/**
 * Run-Length Encoding. Allows to generate and emit RLE chunks.
 */
class RLE(val capacity: Int = 7) {
    /** RAW data in chunks of three values: start, count and data */
    val data = IntArrayList(capacity)

    /** Number of RLE chunks */
    val count get() = data.size / 3

    /** Emits an RLE chunk starting in [start] with [count] elements repeated of the specified [value] */
    fun emit(start: Int, count: Int, value: Int) {
        this.data.add(start, count, value)
    }
    /** Resets this RLE to start from scratch */
    fun clear() {
        data.clear()
    }

    /** Iterates over the RLE chunks emitting in function [block] the number of the chunk, the start of the chunk, its size, and the repeated value */
    inline fun fastForEach(block: (n: Int, start: Int, count: Int, value: Int) -> Unit) {
        for (n in 0 until count) {
            block(n, data[n * 3 + 0], data[n * 3 + 1], data[n * 3 + 2])
        }
    }

    override fun toString(): String = buildString {
        append("RLE(")
        fastForEach { n, start, end, value ->
            if (n != 0) append(", ")
            append("[")
            append("(")
            append(value)
            append("),")
            append(start)
            append(",")
            append(end)
            append("]")
        }
        append(")")
    }

    companion object {
        /** Generates a [RLE] sequence based on the [data] in the segment [start] and [end]. When providing [out], that RLE is reused. */
        inline fun compute(data: IntArray, start: Int = 0, end: Int = data.size, out: RLE = RLE()): RLE =
            compute(end - start, out) { data[start + it] }

        /** Generates a new [RLE] or reuses [out], from a sequence of [count] items, generated by the [gen] function. Emitted value can be filtered with the [filter] function.  */
        inline fun compute(count: Int, out: RLE = RLE(), filter: (value: Int) -> Boolean = { true }, gen: (index: Int) -> Int): RLE {
            out.clear()
            emit(count, emit = { _, start, count, value ->
                if (filter(value)) out.emit(start, count, value)
            }, gen = gen)
            return out
        }

        /** Processes a RLE sequence of [count] elements generated by the [gen] function, and calls [emit] function for every [RLE] segment */
        inline fun emit(count: Int, emit: (n: Int, start: Int, count: Int, value: Int) -> Unit, gen: (index: Int) -> Int) {
            var lastValue = 0
            var currentStart = 0
            var nchunk = 0
            for (n in 0 until count + 1) {
                val value = if (n == count) lastValue + 1 else gen(n)
                if (n == 0 || value != lastValue) {
                    if (currentStart != n) {
                        emit(nchunk++, currentStart, n - currentStart, lastValue)
                    }
                    currentStart = n
                }
                lastValue = value
            }
        }
    }
}

class RLEComparer<T : RLEComparer.Rle>(
    val rlePool: Pool<T>,
    val doDebug: Boolean = false,
) {
    open class Rle(
        var left: Int = 0,
        var right: Int = 0,
    ) {
        fun setTo(left: Int, right: Int) {
            this.left = left
            this.right = right
        }
    }

    fun <T : Rle> T.intersectsWith(that: T): Boolean = (this.left <= that.right) and (this.right >= that.left)
    fun <T : Rle> T.intersections(that: List<T>): List<T> = that.filter { intersectsWith(it) }

    interface Ops<T : Rle> {
        fun zeroToOne(y: Int, nextRle: T): Unit
        fun manyToOne(y: Int, prevRles: List<T>, nextRle: T): Unit
        fun oneToZero(y: Int, prevRle: T): Unit
        fun oneToOne(y: Int, prevRle: T, nextRle: T): Unit
        fun oneToMany(y: Int, prevRle: T, nextRles: List<T>): Unit
    }

    inline fun debug(message: () -> String) {
        if (doDebug) {
            println(message())
        }
    }

    fun compare(
        ops: Ops<T>,
        width: Int,
        height: Int,
        get: (x: Int, y: Int) -> Boolean,
    ) {
        var prevRles = FastArrayList<T>()

        fun RLE.toMy() = FastArrayList<T>().apply {
            fastForEach { n, start, count, value -> add(rlePool.alloc().also { it.setTo(start, start + count) }) }
        }

        for (y in 0 until height) {
            val nextRles = RLE.compute(width, filter = { it != 0 }) { x -> if (get(x, y)) 1 else 0 }.toMy()

            debug { "$nextRles" }

            for (nextRle in nextRles) {
                val intersections = nextRle.intersections(prevRles)

                when {
                    // 0 -> 1
                    intersections.isEmpty() -> {
                        debug { "0 -> 1" }
                        ops.zeroToOne(y, nextRle)
                    }
                    // N -> 1
                    intersections.size >= 2 -> {
                        debug { "N -> 1 :: $intersections" }
                        ops.manyToOne(y, intersections, nextRle)
                    }
                }
            }

            for (prevRle in prevRles) {
                val intersections = prevRle.intersections(nextRles)

                when {
                    // 1 -> 0
                    intersections.isEmpty() -> {
                        debug { "1 -> 0" }
                        ops.oneToZero(y, prevRle)
                    }
                    //// 1 -> 1
                    intersections.size == 1 -> {
                        val nextRle = intersections.first()
                        if (nextRle.intersections(prevRles).size == 1) {
                            debug { "1 -> 1" }
                            ops.oneToOne(y, prevRle, nextRle)
                        }
                    }
                    // 1 -> N
                    else -> {
                        debug { "1 -> N" }
                        ops.oneToMany(y, prevRle, intersections)
                    }
                }
            }

            rlePool.free(prevRles)
            prevRles = nextRles
        }
    }
}
