/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.google.cloud.pubsub.client;

import com.google.common.util.concurrent.MoreExecutors;
import com.spotify.google.cloud.pubsub.client.Message;
import com.spotify.google.cloud.pubsub.client.Pubsub;
import com.spotify.google.cloud.pubsub.client.PubsubException;
import com.spotify.google.cloud.pubsub.client.PubsubFuture;
import com.spotify.google.cloud.pubsub.client.QueueFullException;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Publisher
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(Publisher.class);
    private final Pubsub pubsub;
    private final String project;
    private final int queueSize;
    private final int batchSize;
    private final int concurrency;
    private final long maxLatencyMs;
    private final Listener listener;
    private final AtomicInteger outstanding = new AtomicInteger();
    private final ConcurrentLinkedQueue<TopicQueue> pendingTopics = new ConcurrentLinkedQueue();
    private final ConcurrentMap<String, TopicQueue> topics = new ConcurrentHashMap<String, TopicQueue>();
    private final CompletableFuture<Void> closeFuture = new CompletableFuture();
    private final ScheduledExecutorService scheduler = MoreExecutors.getExitingScheduledExecutorService((ScheduledThreadPoolExecutor)new ScheduledThreadPoolExecutor(1));
    private final AtomicInteger schedulerQueueSize = new AtomicInteger();

    private Publisher(Builder builder) {
        this.pubsub = Objects.requireNonNull(builder.pubsub, "pubsub");
        this.project = Objects.requireNonNull(builder.project, "project");
        this.concurrency = builder.concurrency;
        this.batchSize = builder.batchSize;
        this.queueSize = Optional.ofNullable(builder.queueSize).orElseGet(() -> this.batchSize * 10);
        this.maxLatencyMs = builder.maxLatencyMs;
        this.listener = builder.listener == null ? new ListenerAdapter() : builder.listener;
        this.listener.publisherCreated(this);
    }

    public CompletableFuture<String> publish(String topic, Message message) {
        TopicQueue queue = this.topics.computeIfAbsent(topic, x$0 -> new TopicQueue((String)x$0));
        CompletableFuture future = queue.send(message);
        this.listener.publishingMessage(this, topic, message, future);
        return future;
    }

    @Override
    public void close() {
        this.scheduler.shutdownNow();
        try {
            this.scheduler.awaitTermination(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.closeFuture.complete(null);
        this.listener.publisherClosed(this);
    }

    public CompletableFuture<Void> closeFuture() {
        return this.closeFuture.thenApply(ignore -> null);
    }

    public int outstandingRequests() {
        return this.outstanding.get();
    }

    public int scheduledQueueSize() {
        return this.schedulerQueueSize.get();
    }

    public int pendingTopics() {
        return this.pendingTopics.size();
    }

    public String project() {
        return this.project;
    }

    public int concurrency() {
        return this.concurrency;
    }

    public int topicQueueSize(String topic) {
        TopicQueue topicQueue = (TopicQueue)this.topics.get(topic);
        return topicQueue == null ? 0 : topicQueue.queue.size();
    }

    public int queueSize() {
        return this.queueSize;
    }

    public int batchSize() {
        return this.batchSize;
    }

    private void sendPending() {
        while (this.outstanding.get() < this.concurrency) {
            TopicQueue queue = this.pendingTopics.poll();
            if (queue == null) {
                return;
            }
            queue.pending = false;
            int sent = queue.sendBatch();
            if (sent != this.batchSize) continue;
            queue.pending = true;
            this.pendingTopics.offer(queue);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class ListenerAdapter
    implements Listener {
        @Override
        public void publisherCreated(Publisher publisher) {
        }

        @Override
        public void publisherClosed(Publisher publisher) {
        }

        @Override
        public void publishingMessage(Publisher publisher, String topic, Message message, CompletableFuture<String> future) {
        }

        @Override
        public void sendingBatch(Publisher publisher, String topic, List<Message> batch, PubsubFuture<List<String>> future) {
        }

        @Override
        public void sendingBatch(Publisher publisher, String topic, List<Message> batch, CompletableFuture<List<String>> future) {
        }

        @Override
        public void topicPending(Publisher publisher, String topic, int outstanding, int concurrency) {
        }
    }

    public static class Builder {
        private Pubsub pubsub;
        private String project;
        private Integer queueSize;
        private int batchSize = 1000;
        private int concurrency = 64;
        private Listener listener;
        private long maxLatencyMs = 100L;

        public Builder pubsub(Pubsub pubsub) {
            this.pubsub = pubsub;
            return this;
        }

        public Builder project(String project) {
            this.project = project;
            return this;
        }

        public Builder batchSize(int batchSize) {
            this.batchSize = batchSize;
            return this;
        }

        public Builder queueSize(Integer queueSize) {
            this.queueSize = queueSize;
            return this;
        }

        public Builder concurrency(int concurrency) {
            this.concurrency = concurrency;
            return this;
        }

        public Builder maxLatencyMs(long maxLatencyMs) {
            this.maxLatencyMs = maxLatencyMs;
            return this;
        }

        public Builder listener(Listener listener) {
            this.listener = listener;
            return this;
        }

        public Publisher build() {
            return new Publisher(this);
        }
    }

    private static class QueuedMessage {
        private final Message message;
        private final CompletableFuture<String> future;

        public QueuedMessage(Message message, CompletableFuture<String> future) {
            this.message = message;
            this.future = future;
        }
    }

    private class TopicQueue {
        private final AtomicInteger size = new AtomicInteger();
        private final ConcurrentLinkedQueue<QueuedMessage> queue = new ConcurrentLinkedQueue();
        private final String topic;
        private volatile boolean pending;
        private final AtomicBoolean scheduled = new AtomicBoolean();

        private TopicQueue(String topic) {
            this.topic = topic;
        }

        private CompletableFuture<String> send(Message message) {
            int newSize;
            int currentSize;
            CompletableFuture<String> future = new CompletableFuture<String>();
            do {
                if ((newSize = (currentSize = this.size.get()) + 1) <= Publisher.this.queueSize) continue;
                future.completeExceptionally(new QueueFullException());
                return future;
            } while (!this.size.compareAndSet(currentSize, newSize));
            this.queue.add(new QueuedMessage(message, future));
            this.scheduleSend(newSize);
            return future;
        }

        private void scheduleSend(int queueSize) {
            if (this.pending) {
                return;
            }
            if (queueSize >= Publisher.this.batchSize) {
                this.enqueueSend();
                return;
            }
            if (this.scheduled.compareAndSet(false, true)) {
                try {
                    Publisher.this.scheduler.schedule(this::scheduledEnqueueSend, Publisher.this.maxLatencyMs, TimeUnit.MILLISECONDS);
                    Publisher.this.schedulerQueueSize.incrementAndGet();
                }
                catch (RejectedExecutionException rejectedExecutionException) {
                    // empty catch block
                }
            }
        }

        private void scheduledEnqueueSend() {
            Publisher.this.schedulerQueueSize.decrementAndGet();
            this.scheduled.set(false);
            this.enqueueSendWithErrorLogging();
        }

        private void enqueueSendWithErrorLogging() {
            try {
                this.enqueueSend();
            }
            catch (Exception e) {
                log.error("Error while enqueueing or sending messages on background thread", (Throwable)e);
            }
        }

        private void enqueueSend() {
            int currentOutstanding = Publisher.this.outstanding.get();
            if (currentOutstanding < Publisher.this.concurrency) {
                this.sendBatch();
                return;
            }
            this.pending = true;
            Publisher.this.pendingTopics.offer(this);
            Publisher.this.listener.topicPending(Publisher.this, this.topic, currentOutstanding, Publisher.this.concurrency);
            Publisher.this.sendPending();
        }

        private int sendBatch() {
            QueuedMessage message;
            ArrayList<Message> batch = new ArrayList<Message>();
            ArrayList<CompletableFuture> futures = new ArrayList<CompletableFuture>();
            while (batch.size() < Publisher.this.batchSize && (message = this.queue.poll()) != null) {
                batch.add(message.message);
                futures.add(message.future);
            }
            if (batch.size() == 0) {
                return 0;
            }
            this.size.updateAndGet(i -> i - batch.size());
            Publisher.this.outstanding.incrementAndGet();
            PubsubFuture<List<String>> batchFuture = Publisher.this.pubsub.publish(Publisher.this.project, this.topic, batch);
            Publisher.this.listener.sendingBatch(Publisher.this, this.topic, Collections.unmodifiableList(batch), batchFuture);
            ((PubsubFuture)batchFuture.whenComplete((messageIds, ex) -> {
                Publisher.this.outstanding.decrementAndGet();
                if (ex != null) {
                    futures.forEach(f -> f.completeExceptionally((Throwable)ex));
                    return;
                }
                if (futures.size() != messageIds.size()) {
                    futures.forEach(f -> f.completeExceptionally(new PubsubException("message id count mismatch: " + futures.size() + " != " + messageIds.size())));
                }
                for (int i = 0; i < futures.size(); ++i) {
                    String messageId = (String)messageIds.get(i);
                    CompletableFuture future = (CompletableFuture)futures.get(i);
                    future.complete(messageId);
                }
            })).whenComplete((v, t) -> Publisher.this.sendPending());
            return batch.size();
        }
    }

    public static interface Listener {
        public void publisherCreated(Publisher var1);

        public void publisherClosed(Publisher var1);

        public void publishingMessage(Publisher var1, String var2, Message var3, CompletableFuture<String> var4);

        @Deprecated
        public void sendingBatch(Publisher var1, String var2, List<Message> var3, CompletableFuture<List<String>> var4);

        default public void sendingBatch(Publisher publisher, String topic, List<Message> batch, PubsubFuture<List<String>> future) {
            this.sendingBatch(publisher, topic, batch, (CompletableFuture<List<String>>)future);
        }

        public void topicPending(Publisher var1, String var2, int var3, int var4);
    }
}

