/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.docker.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.spotify.docker.client.OsUtils;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.text.MessageFormat;
import java.util.EnumSet;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CompressedDirectory
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(CompressedDirectory.class);
    private static final int DEFAULT_FILE_MODE = 33188;
    private static final String POSIX_FILE_VIEW = "posix";
    private final Path file;

    private CompressedDirectory(Path file) {
        this.file = file;
    }

    public Path file() {
        return this.file;
    }

    public static CompressedDirectory create(Path directory) throws IOException {
        Path file = Files.createTempFile("docker-client-", ".tar.gz", new FileAttribute[0]);
        Path dockerIgnorePath = directory.resolve(".dockerignore");
        ImmutableList<DockerIgnorePathMatcher> ignoreMatchers = CompressedDirectory.parseDockerIgnore(dockerIgnorePath);
        try (OutputStream fileOut = Files.newOutputStream(file, new OpenOption[0]);
             GzipCompressorOutputStream gzipOut = new GzipCompressorOutputStream(fileOut);
             TarArchiveOutputStream tarOut = new TarArchiveOutputStream((OutputStream)gzipOut);){
            tarOut.setLongFileMode(3);
            tarOut.setBigNumberMode(2);
            Files.walkFileTree(directory, EnumSet.of(FileVisitOption.FOLLOW_LINKS), Integer.MAX_VALUE, new Visitor(directory, ignoreMatchers, tarOut));
        }
        catch (Throwable t) {
            try {
                Files.delete(file);
            }
            catch (IOException e) {
                t.addSuppressed(e);
            }
            throw t;
        }
        return new CompressedDirectory(file);
    }

    @Override
    public void close() throws IOException {
        Files.delete(this.file);
    }

    static ImmutableList<DockerIgnorePathMatcher> parseDockerIgnore(Path dockerIgnorePath) throws IOException {
        ImmutableList.Builder matchersBuilder = ImmutableList.builder();
        if (Files.isReadable(dockerIgnorePath) && Files.isRegularFile(dockerIgnorePath, new LinkOption[0])) {
            for (String line : Files.readAllLines(dockerIgnorePath, StandardCharsets.UTF_8)) {
                String pattern = CompressedDirectory.createPattern(line);
                if (Strings.isNullOrEmpty((String)pattern)) {
                    log.debug("Will skip '{}' - because it's empty after trimming or it's a comment", (Object)line);
                    continue;
                }
                if (pattern.startsWith("!")) {
                    matchersBuilder.add((Object)new DockerIgnorePathMatcher(dockerIgnorePath.getFileSystem(), pattern, false));
                    continue;
                }
                matchersBuilder.add((Object)new DockerIgnorePathMatcher(dockerIgnorePath.getFileSystem(), pattern, true));
            }
        }
        return matchersBuilder.build();
    }

    private static String createPattern(String line) {
        String pattern = line.trim();
        if (pattern.startsWith("#")) {
            return null;
        }
        if (OsUtils.isLinux() || OsUtils.isOsX()) {
            return pattern;
        }
        return pattern.replace("/", "\\\\");
    }

    @VisibleForTesting
    static PathMatcher goPathMatcher(FileSystem fs, String pattern) {
        String notSeparatorPattern = CompressedDirectory.getNotSeparatorPattern(fs.getSeparator());
        String starPattern = String.format("%s*", notSeparatorPattern);
        StringBuilder patternBuilder = new StringBuilder();
        boolean inCharRange = false;
        boolean inEscape = false;
        block10: for (int i = 0; i < pattern.length(); ++i) {
            char c = pattern.charAt(i);
            if (inCharRange) {
                if (inEscape) {
                    patternBuilder.append(c);
                    inEscape = false;
                    continue;
                }
                switch (c) {
                    case '\\': {
                        patternBuilder.append('\\');
                        inEscape = true;
                        break;
                    }
                    case ']': {
                        patternBuilder.append(']');
                        inCharRange = false;
                        break;
                    }
                    default: {
                        patternBuilder.append(c);
                        break;
                    }
                }
                continue;
            }
            if (inEscape) {
                patternBuilder.append(Pattern.quote(Character.toString(c)));
                inEscape = false;
                continue;
            }
            switch (c) {
                case '*': {
                    patternBuilder.append(starPattern);
                    continue block10;
                }
                case '?': {
                    patternBuilder.append(notSeparatorPattern);
                    continue block10;
                }
                case '[': {
                    patternBuilder.append("[");
                    inCharRange = true;
                    continue block10;
                }
                case '\\': {
                    inEscape = true;
                    continue block10;
                }
                default: {
                    patternBuilder.append(Pattern.quote(Character.toString(c)));
                }
            }
        }
        return fs.getPathMatcher("regex:" + patternBuilder.toString());
    }

    private static String getNotSeparatorPattern(String separator) {
        switch (separator) {
            case "/": {
                return "[^/]";
            }
            case "\\": {
                return "[^\\\\]";
            }
        }
        String message = MessageFormat.format("Filepath matching not supported for file system separator {0}", separator);
        throw new UnsupportedOperationException(message);
    }

    private static class DockerIgnorePathMatcher
    implements PathMatcher {
        private final String pattern;
        private final PathMatcher matcher;
        private final boolean exclude;

        public DockerIgnorePathMatcher(FileSystem fileSystem, String pattern, boolean exclude) {
            this.exclude = exclude;
            this.pattern = pattern;
            this.matcher = exclude ? CompressedDirectory.goPathMatcher(fileSystem, pattern) : CompressedDirectory.goPathMatcher(fileSystem, pattern.substring(1));
        }

        public boolean isExclude() {
            return this.exclude;
        }

        @Override
        public boolean matches(Path path) {
            return path.startsWith(this.pattern) || this.matcher.matches(path);
        }

        public String toString() {
            return this.pattern;
        }
    }

    private static class Visitor
    extends SimpleFileVisitor<Path> {
        private final Path root;
        private final ImmutableList<DockerIgnorePathMatcher> ignoreMatchers;
        private final TarArchiveOutputStream tarStream;

        private Visitor(Path root, ImmutableList<DockerIgnorePathMatcher> ignoreMatchers, TarArchiveOutputStream tarStream) {
            this.root = root;
            this.ignoreMatchers = ignoreMatchers.reverse();
            this.tarStream = tarStream;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Path relativePath = this.root.relativize(file);
            if (Visitor.exclude(this.ignoreMatchers, relativePath)) {
                return FileVisitResult.CONTINUE;
            }
            TarArchiveEntry entry = new TarArchiveEntry(file.toFile());
            entry.setName(relativePath.toString());
            entry.setMode(Visitor.getFileMode(file));
            entry.setSize(attrs.size());
            this.tarStream.putArchiveEntry((ArchiveEntry)entry);
            Files.copy(file, (OutputStream)this.tarStream);
            this.tarStream.closeArchiveEntry();
            return FileVisitResult.CONTINUE;
        }

        private static boolean exclude(ImmutableList<DockerIgnorePathMatcher> matchers, Path path) {
            for (DockerIgnorePathMatcher matcher : matchers) {
                if (!matcher.matches(path)) continue;
                return matcher.isExclude();
            }
            return false;
        }

        private static int getFileMode(Path file) throws IOException {
            if (Visitor.isPosixComplantFs()) {
                return Visitor.getPosixFileMode(file);
            }
            return 33188;
        }

        private static boolean isPosixComplantFs() {
            return FileSystems.getDefault().supportedFileAttributeViews().contains(CompressedDirectory.POSIX_FILE_VIEW);
        }

        private static int getPosixFileMode(Path file) throws IOException {
            PosixFileAttributes attr = Files.readAttributes(file, PosixFileAttributes.class, new LinkOption[0]);
            Set<PosixFilePermission> perm = attr.permissions();
            int mode = 32768;
            mode += 64 * Visitor.getModeFromPermissions(perm.contains((Object)PosixFilePermission.OWNER_READ), perm.contains((Object)PosixFilePermission.OWNER_WRITE), perm.contains((Object)PosixFilePermission.OWNER_EXECUTE));
            mode += 8 * Visitor.getModeFromPermissions(perm.contains((Object)PosixFilePermission.GROUP_READ), perm.contains((Object)PosixFilePermission.GROUP_WRITE), perm.contains((Object)PosixFilePermission.GROUP_EXECUTE));
            return mode += Visitor.getModeFromPermissions(perm.contains((Object)PosixFilePermission.OTHERS_READ), perm.contains((Object)PosixFilePermission.OTHERS_WRITE), perm.contains((Object)PosixFilePermission.OTHERS_EXECUTE));
        }

        private static int getModeFromPermissions(boolean read, boolean write, boolean execute) {
            int result = 0;
            if (read) {
                result += 4;
            }
            if (write) {
                result += 2;
            }
            if (execute) {
                ++result;
            }
            return result;
        }
    }
}

