/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.flo.context;

import com.spotify.flo.Fn;
import com.spotify.flo.freezer.PersistingContext;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.lang.management.ManagementFactory;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ForkingExecutor
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(ForkingExecutor.class);
    private final List<Execution> executions = new ArrayList<Execution>();
    private Map<String, String> environment = Collections.emptyMap();
    private List<String> javaArgs = Collections.emptyList();

    ForkingExecutor() {
    }

    ForkingExecutor environment(Map<String, String> environment) {
        this.environment = new HashMap<String, String>(environment);
        return this;
    }

    ForkingExecutor javaArgs(String ... javaArgs) {
        return this.javaArgs(Arrays.asList(javaArgs));
    }

    ForkingExecutor javaArgs(List<String> javaArgs) {
        this.javaArgs = new ArrayList<String>(javaArgs);
        return this;
    }

    <T> T execute(Fn<T> f) throws IOException {
        try (Execution<T> execution = new Execution<T>(f);){
            this.executions.add(execution);
            execution.start();
            T t = execution.waitFor();
            return t;
        }
    }

    @Override
    public void close() {
        this.executions.forEach(Execution::close);
    }

    private static void tryDeleteDir(Path path) {
        try {
            ForkingExecutor.deleteDir(path);
        }
        catch (IOException e) {
            log.warn("Failed to delete directory: {}", (Object)path, (Object)e);
        }
    }

    private static void deleteDir(Path path) throws IOException {
        try {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    try {
                        Files.delete(file);
                    }
                    catch (NoSuchFileException noSuchFileException) {
                        // empty catch block
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    try {
                        Files.delete(dir);
                    }
                    catch (NoSuchFileException noSuchFileException) {
                        // empty catch block
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (NoSuchFileException noSuchFileException) {
            // empty catch block
        }
    }

    private static void copyLines(InputStream in, PrintStream out) {
        BufferedReader reader = new BufferedReader(new InputStreamReader(in));
        try {
            String line;
            while ((line = reader.readLine()) != null) {
                out.println(line);
            }
        }
        catch (IOException e) {
            log.error("Caught exception during stream copy", (Throwable)e);
        }
    }

    private static class Trampoline {
        private Trampoline() {
        }

        public static void main(String ... args) {
            Path errorFile;
            Path resultFile;
            Path closureFile;
            log.debug("child process started: args={}", Arrays.asList(args));
            Watchdog watchdog = new Watchdog();
            watchdog.start();
            if (args.length != 3) {
                log.error("args.length != 3");
                System.exit(3);
                return;
            }
            try {
                closureFile = Paths.get(args[0], new String[0]);
                resultFile = Paths.get(args[1], new String[0]);
                errorFile = Paths.get(args[2], new String[0]);
            }
            catch (InvalidPathException e) {
                log.error("Failed to get file path", (Throwable)e);
                System.exit(4);
                return;
            }
            Trampoline.run(closureFile, resultFile, errorFile);
        }

        private static void run(Path closureFile, Path resultFile, Path errorFile) {
            Fn fn;
            log.debug("deserializing closure: {}", (Object)closureFile);
            try {
                fn = (Fn)PersistingContext.deserialize((Path)closureFile);
            }
            catch (Exception e) {
                log.error("Failed to deserialize closure: {}", (Object)closureFile, (Object)e);
                System.exit(5);
                return;
            }
            log.debug("executing closure");
            Object result = null;
            Exception error = null;
            try {
                result = fn.get();
            }
            catch (Exception e) {
                error = e;
            }
            if (error != null) {
                log.debug("serializing error", (Throwable)error);
                try {
                    PersistingContext.serialize((Object)error, (Path)errorFile);
                }
                catch (Exception e) {
                    log.error("failed to serialize error", (Throwable)e);
                    System.exit(6);
                    return;
                }
            }
            log.debug("serializing result: {}", result);
            try {
                PersistingContext.serialize((Object)result, (Path)resultFile);
            }
            catch (Exception e) {
                log.error("failed to serialize result", (Throwable)e);
                System.exit(7);
                return;
            }
            System.err.flush();
            System.exit(0);
        }

        private static class Watchdog
        extends Thread {
            Watchdog() {
                this.setDaemon(false);
            }

            @Override
            public void run() {
                try {
                    int c;
                    while ((c = System.in.read()) != -1) {
                    }
                }
                catch (IOException e) {
                    log.error("watchdog failed", (Throwable)e);
                }
                log.debug("child process exiting");
                System.exit(-1);
            }
        }
    }

    private class Execution<T>
    implements Closeable {
        private final ExecutorService executor = Executors.newCachedThreadPool();
        private final Path tempdir = Files.createTempDirectory("flo-fork", new FileAttribute[0]);
        private final Path workdir = Files.createDirectory(this.tempdir.resolve("workdir"), new FileAttribute[0]);
        private final Path closureFile = this.tempdir.resolve("closure");
        private final Path resultFile = this.tempdir.resolve("result");
        private final Path errorFile = this.tempdir.resolve("error");
        private final String home = System.getProperty("java.home");
        private final String classPath = System.getProperty("java.class.path");
        private final Path java = Paths.get(this.home, "bin", "java").toAbsolutePath().normalize();
        private final Fn<T> f;
        private Process process;

        Execution(Fn<T> f) throws IOException {
            this.f = Objects.requireNonNull(f);
        }

        void start() {
            if (this.process != null) {
                throw new IllegalStateException();
            }
            log.debug("serializing closure");
            try {
                PersistingContext.serialize(this.f, (Path)this.closureFile);
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to serialize closure", e);
            }
            ProcessBuilder processBuilder = new ProcessBuilder(this.java.toString(), "-cp", this.classPath).directory(this.workdir.toFile());
            ManagementFactory.getRuntimeMXBean().getInputArguments().stream().filter(s -> s.startsWith("-Xmx") || s.startsWith("-D")).forEach(processBuilder.command()::add);
            ForkingExecutor.this.javaArgs.forEach(processBuilder.command()::add);
            processBuilder.command().add(Trampoline.class.getName());
            processBuilder.command().add(this.closureFile.toString());
            processBuilder.command().add(this.resultFile.toString());
            processBuilder.command().add(this.errorFile.toString());
            processBuilder.environment().putAll(ForkingExecutor.this.environment);
            log.debug("Starting subprocess: environment={}, command={}, directory={}", new Object[]{processBuilder.environment(), processBuilder.command(), processBuilder.directory()});
            try {
                this.process = processBuilder.start();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            this.executor.submit(() -> ForkingExecutor.copyLines(this.process.getInputStream(), System.out));
            this.executor.submit(() -> ForkingExecutor.copyLines(this.process.getErrorStream(), System.err));
        }

        T waitFor() {
            Object result;
            int exitValue;
            if (this.process == null) {
                throw new IllegalStateException();
            }
            log.debug("Waiting for subprocess exit");
            try {
                exitValue = this.process.waitFor();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
            finally {
                this.process.destroyForcibly();
            }
            log.debug("Subprocess exited: " + exitValue);
            if (exitValue != 0) {
                throw new RuntimeException("Subprocess failed: " + this.process.exitValue());
            }
            if (Files.exists(this.errorFile, new LinkOption[0])) {
                Throwable error;
                log.debug("Subprocess exited with error file");
                try {
                    error = (Throwable)PersistingContext.deserialize((Path)this.errorFile);
                }
                catch (Exception e) {
                    throw new RuntimeException("Failed to deserialize error", e);
                }
                if (error instanceof RuntimeException) {
                    throw (RuntimeException)error;
                }
                throw new RuntimeException(error);
            }
            log.debug("Subprocess exited with result file");
            try {
                result = PersistingContext.deserialize((Path)this.resultFile);
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to deserialize result", e);
            }
            return (T)result;
        }

        @Override
        public void close() {
            if (this.process != null) {
                this.process.destroyForcibly();
                this.process = null;
            }
            this.executor.shutdown();
            ForkingExecutor.tryDeleteDir(this.tempdir);
        }
    }
}

