/*
 * Copyright (c) 2015 Spotify AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.spotify.folsom;

import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.ListenableFuture;

public class ConnectFuture
        extends AbstractFuture<Void>
        implements ConnectionChangeListener {

  private final boolean awaitedState;

  /**
   * Create a future that completes once the client reaches the awaited state
   * @param client
   * @param awaitedState
   */
  private ConnectFuture(RawMemcacheClient client, boolean awaitedState) {
    this.awaitedState = awaitedState;
    client.registerForConnectionChanges(this);
    check(client);
  }

  public static ListenableFuture<Void> disconnectFuture(RawMemcacheClient client) {
    return new ConnectFuture(client, false);
  }

  public static ListenableFuture<Void> connectFuture(RawMemcacheClient client) {
    return new ConnectFuture(client, true);
  }

  @Override
  public void connectionChanged(RawMemcacheClient client) {
    check(client);
  }

  private void check(RawMemcacheClient client) {
    if (awaitedState == client.isConnected()) {
      if (set(null)) {
        client.unregisterForConnectionChanges(this);
      }
    }
  }
}
