/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.common;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.spotify.helios.common.descriptors.Job;
import com.spotify.helios.common.descriptors.JobId;
import com.spotify.helios.common.descriptors.PortMapping;
import com.spotify.helios.common.descriptors.ServiceEndpoint;
import com.spotify.helios.common.descriptors.ServicePorts;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

public class JobValidator {
    public static final Pattern NAME_VERSION_PATTERN = Pattern.compile("[0-9a-zA-Z-_.]+");
    public static final Pattern DOMAIN_PATTERN = Pattern.compile("^(?:(?:[a-zA-Z0-9]|(?:[a-zA-Z0-9][a-zA-Z0-9\\-]*[a-zA-Z0-9]))(\\.(?:[a-zA-Z0-9]|(?:[a-zA-Z0-9][a-zA-Z0-9\\-]*[a-zA-Z0-9])))*)\\.?$");
    public static final Pattern IPV4_PATTERN = Pattern.compile("^(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$");
    public static final Pattern NAMESPACE_PATTERN = Pattern.compile("^([a-z0-9_]{4,30})$");
    public static final Pattern REPO_PATTERN = Pattern.compile("^([a-z0-9-_.]+)$");
    public static final Pattern DIGIT_PERIOD = Pattern.compile("^[0-9.]+$");
    public static final Pattern PORT_MAPPING_PROTO_PATTERN = Pattern.compile("(tcp|udp)");
    public static final Pattern PORT_MAPPING_NAME_PATTERN = Pattern.compile("\\S+");
    public static final Pattern REGISTRATION_NAME_PATTERN = Pattern.compile("[_\\-\\w]+");

    public Set<String> validate(Job job) {
        HashSet errors = Sets.newHashSet();
        errors.addAll(this.validateJobId(job));
        errors.addAll(this.validateJobImage(job.getImage()));
        HashSet externalPorts = Sets.newHashSet();
        for (PortMapping portMapping : job.getPorts().values()) {
            Integer externalMappedPort = portMapping.getExternalPort();
            if (externalPorts.contains(externalMappedPort) && externalMappedPort != null) {
                errors.add(String.format("Duplicate external port mapping: %s", externalMappedPort));
            }
            externalPorts.add(externalMappedPort);
        }
        for (Map.Entry entry : job.getPorts().entrySet()) {
            String name = (String)entry.getKey();
            PortMapping mapping = (PortMapping)entry.getValue();
            if (!PORT_MAPPING_PROTO_PATTERN.matcher(mapping.getProtocol()).matches()) {
                errors.add(String.format("Invalid port mapping protocol: %s", mapping.getProtocol()));
            }
            if (!this.legalPort(mapping.getInternalPort())) {
                errors.add(String.format("Invalid internal port: %d", mapping.getInternalPort()));
            }
            if (mapping.getExternalPort() != null && !this.legalPort(mapping.getExternalPort())) {
                errors.add(String.format("Invalid external port: %d", mapping.getExternalPort()));
            }
            if (PORT_MAPPING_NAME_PATTERN.matcher(name).matches()) continue;
            errors.add(String.format("Invalid port mapping endpoint name: %s", name));
        }
        for (ServiceEndpoint serviceEndpoint : job.getRegistration().keySet()) {
            ServicePorts servicePorts = job.getRegistration().get(serviceEndpoint);
            for (String portName : servicePorts.getPorts().keySet()) {
                if (!job.getPorts().containsKey(portName)) {
                    errors.add(String.format("Service registration refers to missing port mapping: %s=%s", serviceEndpoint, portName));
                }
                if (REGISTRATION_NAME_PATTERN.matcher(serviceEndpoint.getName()).matches()) continue;
                errors.add(String.format("Invalid service registration name: %s", serviceEndpoint.getName()));
            }
        }
        for (Map.Entry entry : job.getVolumes().entrySet()) {
            String path = (String)entry.getKey();
            String source = (String)entry.getValue();
            if (!path.startsWith("/")) {
                errors.add("Volume path is not absolute: " + path);
                continue;
            }
            if (!Strings.isNullOrEmpty((String)source) && !source.startsWith("/")) {
                errors.add("Volume source is not absolute: " + source);
                continue;
            }
            String[] parts = path.split(":", 3);
            if (!path.isEmpty() && !path.equals("/") && parts.length <= 2 && (parts.length <= 1 || !parts[1].isEmpty())) continue;
            errors.add(String.format("Invalid volume path: %s", path));
        }
        return errors;
    }

    private Set<String> validateJobImage(String image) {
        HashSet errors = Sets.newHashSet();
        if (image == null) {
            errors.add(String.format("Image was not specified.", new Object[0]));
        } else {
            if (image.endsWith(":latest")) {
                errors.add("Cannot use images that are tagged with :latest, use the hex id instead");
            }
            this.validateImageReference(image, errors);
        }
        return errors;
    }

    private Set<String> validateJobId(Job job) {
        HashSet errors = Sets.newHashSet();
        JobId jobId = job.getId();
        String jobIdVersion = jobId.getVersion();
        String jobIdHash = jobId.getHash();
        JobId recomputedId = job.toBuilder().build().getId();
        if (jobId == null) {
            errors.add(String.format("Job id was not specified.", new Object[0]));
            return errors;
        }
        errors.addAll(this.validateJobName(jobId, recomputedId));
        errors.addAll(this.validateJobVersion(jobIdVersion, recomputedId));
        errors.addAll(this.validateJobHash(jobIdHash, recomputedId));
        return errors;
    }

    private Set<String> validateJobName(JobId jobId, JobId recomputedId) {
        HashSet errors = Sets.newHashSet();
        String jobIdName = jobId.getName();
        if (jobIdName == null || jobIdName.isEmpty()) {
            errors.add(String.format("Job name was not specified.", new Object[0]));
            return errors;
        }
        if (!NAME_VERSION_PATTERN.matcher(jobIdName).matches()) {
            errors.add(String.format("Job name may only contain [0-9a-zA-Z-_.] in job name [%s].", recomputedId.getName()));
        }
        if (!recomputedId.getName().equals(jobIdName)) {
            errors.add(String.format("Id name mismatch: %s != %s", jobIdName, recomputedId.getName()));
        }
        return errors;
    }

    private Set<String> validateJobVersion(String jobIdVersion, JobId recomputedId) {
        HashSet errors = Sets.newHashSet();
        if (jobIdVersion == null || jobIdVersion.isEmpty()) {
            errors.add(String.format("Job version was not specified in job id [%s].", recomputedId));
            return errors;
        }
        if (!NAME_VERSION_PATTERN.matcher(jobIdVersion).matches()) {
            errors.add(String.format("Job version may only contain [0-9a-zA-Z-_.] in job version [%s].", recomputedId.getVersion()));
        }
        if (!recomputedId.getVersion().equals(jobIdVersion)) {
            errors.add(String.format("Id version mismatch: %s != %s", jobIdVersion, recomputedId.getVersion()));
        }
        return errors;
    }

    private Set<String> validateJobHash(String jobIdHash, JobId recomputedId) {
        HashSet errors = Sets.newHashSet();
        if (jobIdHash == null || jobIdHash.isEmpty()) {
            errors.add(String.format("Job hash was not specified in job id [%s].", recomputedId));
            return errors;
        }
        if (jobIdHash.indexOf(58) != -1) {
            errors.add(String.format("Job hash contains colon in job id [%s].", recomputedId));
        }
        if (!recomputedId.getHash().equals(jobIdHash)) {
            errors.add(String.format("Id hash mismatch: %s != %s", jobIdHash, recomputedId.getHash()));
        }
        return errors;
    }

    private boolean validateImageReference(String imageRef, Collection<String> errors) {
        String repo;
        String tag;
        boolean valid = true;
        int lastColon = imageRef.lastIndexOf(58);
        if (lastColon != -1 && !(tag = imageRef.substring(lastColon + 1)).contains("/")) {
            repo = imageRef.substring(0, lastColon);
            valid &= this.validateTag(tag, errors);
        } else {
            repo = imageRef;
        }
        String invalidRepoName = "Invalid repository name (ex: \"registry.domain.tld/myrepos\")";
        if (repo.contains("://")) {
            errors.add("Invalid repository name (ex: \"registry.domain.tld/myrepos\")");
            return false;
        }
        String[] nameParts = repo.split("/", 2);
        if (!(nameParts[0].contains(".") || nameParts[0].contains(":") || nameParts[0].equals("localhost"))) {
            return this.validateRepositoryName(repo, errors);
        }
        if (nameParts.length < 2) {
            errors.add("Invalid repository name (ex: \"registry.domain.tld/myrepos\")");
            return false;
        }
        String endpoint = nameParts[0];
        String reposName = nameParts[1];
        valid &= this.validateEndpoint(endpoint, errors);
        return valid &= this.validateRepositoryName(reposName, errors);
    }

    private boolean validateTag(String tag, Collection<String> errors) {
        boolean valid = true;
        if (tag.isEmpty()) {
            errors.add("Tag cannot be empty");
            valid = false;
        }
        if (tag.contains("/") || tag.contains(":")) {
            errors.add(String.format("Illegal tag: \"%s\"", tag));
            valid = false;
        }
        return valid;
    }

    private boolean validateEndpoint(String endpoint, Collection<String> errors) {
        String[] parts = endpoint.split(":", 2);
        if (!this.validateAddress(parts[0], errors)) {
            return false;
        }
        if (parts.length > 1) {
            int port;
            try {
                port = Integer.valueOf(parts[1]);
            }
            catch (NumberFormatException e) {
                errors.add(String.format("Invalid port in endpoint: \"%s\"", endpoint));
                return false;
            }
            if (port < 0 || port > 65535) {
                errors.add(String.format("Invalid port in endpoint: \"%s\"", endpoint));
                return false;
            }
        }
        return true;
    }

    private boolean validateAddress(String address, Collection<String> errors) {
        if (IPV4_PATTERN.matcher(address).matches()) {
            return true;
        }
        if (!DOMAIN_PATTERN.matcher(address).matches() || DIGIT_PERIOD.matcher(address).find()) {
            errors.add(String.format("Invalid domain name: \"%s\"", address));
            return false;
        }
        return true;
    }

    private boolean validateRepositoryName(String repositoryName, Collection<String> errors) {
        String name;
        String repo;
        boolean valid = true;
        String[] nameParts = repositoryName.split("/", 2);
        if (nameParts.length < 2) {
            repo = "library";
            name = nameParts[0];
        } else {
            repo = nameParts[0];
            name = nameParts[1];
        }
        if (!NAMESPACE_PATTERN.matcher(repo).matches()) {
            errors.add(String.format("Invalid namespace name (%s), only [a-z0-9_] are allowed, size between 4 and 30", repo));
            valid = false;
        }
        if (!REPO_PATTERN.matcher(name).matches()) {
            errors.add(String.format("Invalid repository name (%s), only [a-z0-9-_.] are allowed", name));
            valid = false;
        }
        return valid;
    }

    private boolean legalPort(int port) {
        return port >= 0 && port <= 65535;
    }
}

