/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.common;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import com.spotify.helios.common.descriptors.ExecHealthCheck;
import com.spotify.helios.common.descriptors.HealthCheck;
import com.spotify.helios.common.descriptors.HttpHealthCheck;
import com.spotify.helios.common.descriptors.Job;
import com.spotify.helios.common.descriptors.JobId;
import com.spotify.helios.common.descriptors.PortMapping;
import com.spotify.helios.common.descriptors.ServiceEndpoint;
import com.spotify.helios.common.descriptors.ServicePorts;
import com.spotify.helios.common.descriptors.TcpHealthCheck;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

public class JobValidator {
    private static final Pattern NAME_VERSION_PATTERN = Pattern.compile("[0-9a-zA-Z-_.]+");
    private static final Pattern HOSTNAME_PATTERN = Pattern.compile("^([a-z0-9][a-z0-9-]{0,62}$)");
    private static final Pattern DOMAIN_PATTERN = Pattern.compile("^(?:(?:[a-zA-Z0-9]|(?:[a-zA-Z0-9][a-zA-Z0-9\\-]*[a-zA-Z0-9]))(\\.(?:[a-zA-Z0-9]|(?:[a-zA-Z0-9][a-zA-Z0-9\\-]*[a-zA-Z0-9])))*)\\.?$");
    private static final Pattern IPV4_PATTERN = Pattern.compile("^(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$");
    private static final Pattern NAME_COMPONENT_PATTERN = Pattern.compile("^([a-z0-9._-]+)$");
    private static final int REPO_NAME_MAX_LENGTH = 255;
    private static final Pattern TAG_PATTERN = Pattern.compile("[\\w][\\w.-]{0,127}");
    private static final Pattern DIGIT_PERIOD = Pattern.compile("^[0-9.]+$");
    private static final Pattern PORT_MAPPING_PROTO_PATTERN = Pattern.compile("(tcp|udp)");
    private static final Pattern PORT_MAPPING_NAME_PATTERN = Pattern.compile("\\S+");
    private static final Pattern REGISTRATION_NAME_PATTERN = Pattern.compile("[_\\-\\w]+");
    private static final List<String> VALID_NETWORK_MODES = ImmutableList.of((Object)"bridge", (Object)"host");
    private final boolean shouldValidateJobHash;
    private final boolean shouldValidateAddCapabilities;
    private final Set<String> whitelistedCapabilities;

    public JobValidator() {
        this(true);
    }

    public JobValidator(boolean shouldValidateJobHash) {
        this(shouldValidateJobHash, false);
    }

    public JobValidator(boolean shouldValidateJobHash, boolean shouldValidateAddCapabilities) {
        this(shouldValidateJobHash, shouldValidateAddCapabilities, Collections.emptySet());
    }

    public JobValidator(boolean shouldValidateJobHash, boolean shouldValidateAddCapabilities, Set<String> whitelistedCapabilities) {
        this.shouldValidateJobHash = shouldValidateJobHash;
        this.shouldValidateAddCapabilities = shouldValidateAddCapabilities;
        this.whitelistedCapabilities = whitelistedCapabilities;
    }

    public Set<String> validate(Job job) {
        HashSet errors = Sets.newHashSet();
        errors.addAll(this.validateJobId(job));
        errors.addAll(this.validateJobImage(job.getImage()));
        errors.addAll(this.validateJobHostName(job.getHostname()));
        HashSet externalPorts = Sets.newHashSet();
        for (PortMapping portMapping : job.getPorts().values()) {
            Integer externalMappedPort = portMapping.getExternalPort();
            if (externalPorts.contains(externalMappedPort) && externalMappedPort != null) {
                errors.add(String.format("Duplicate external port mapping: %s", externalMappedPort));
            }
            externalPorts.add(externalMappedPort);
        }
        for (Map.Entry entry : job.getPorts().entrySet()) {
            String name = (String)entry.getKey();
            PortMapping mapping = (PortMapping)entry.getValue();
            if (!PORT_MAPPING_PROTO_PATTERN.matcher(mapping.getProtocol()).matches()) {
                errors.add(String.format("Invalid port mapping protocol: %s", mapping.getProtocol()));
            }
            if (!this.legalPort(mapping.getInternalPort())) {
                errors.add(String.format("Invalid internal port: %d", mapping.getInternalPort()));
            }
            if (mapping.getExternalPort() != null && !this.legalPort(mapping.getExternalPort())) {
                errors.add(String.format("Invalid external port: %d", mapping.getExternalPort()));
            }
            if (PORT_MAPPING_NAME_PATTERN.matcher(name).matches()) continue;
            errors.add(String.format("Invalid port mapping endpoint name: %s", name));
        }
        for (ServiceEndpoint serviceEndpoint : job.getRegistration().keySet()) {
            ServicePorts servicePorts = job.getRegistration().get(serviceEndpoint);
            if (servicePorts == null || servicePorts.getPorts() == null) {
                errors.add(String.format("registration for '%s' is malformed: does not have a port mapping", serviceEndpoint.getName()));
                continue;
            }
            for (String portName : servicePorts.getPorts().keySet()) {
                if (!job.getPorts().containsKey(portName)) {
                    errors.add(String.format("Service registration refers to missing port mapping: %s=%s", serviceEndpoint, portName));
                }
                if (REGISTRATION_NAME_PATTERN.matcher(serviceEndpoint.getName()).matches()) continue;
                errors.add(String.format("Invalid service registration name: %s", serviceEndpoint.getName()));
            }
        }
        for (Map.Entry entry : job.getVolumes().entrySet()) {
            String path = (String)entry.getKey();
            String source = (String)entry.getValue();
            if (!path.startsWith("/")) {
                errors.add("Volume path is not absolute: " + path);
                continue;
            }
            if (!Strings.isNullOrEmpty((String)source) && !source.startsWith("/")) {
                errors.add("Volume source is not absolute: " + source);
                continue;
            }
            String[] parts = path.split(":", 3);
            if (!path.isEmpty() && !(path.equals("/") | parts.length > 2) && (parts.length <= 1 || !parts[1].isEmpty())) continue;
            errors.add(String.format("Invalid volume path: %s", path));
        }
        Date expiry = job.getExpires();
        if (expiry != null && expiry.before(new Date())) {
            errors.add("Job expires in the past");
        }
        errors.addAll(this.validateJobHealthCheck(job));
        errors.addAll(this.validateJobNetworkMode(job));
        if (this.shouldValidateAddCapabilities) {
            errors.addAll(this.validateAddCapabilities(job));
        }
        for (String mountPoint : job.getRamdisks().keySet()) {
            if (mountPoint.startsWith("/")) continue;
            errors.add("Ramdisk mount point is not absolute: " + mountPoint);
        }
        HashSet hashSet = Sets.newHashSet();
        for (String s : job.getVolumes().keySet()) {
            hashSet.add(s.split(":", 2)[0]);
        }
        for (String mountPoint : job.getRamdisks().keySet()) {
            if (!hashSet.contains(mountPoint)) continue;
            errors.add(String.format("Ramdisk mount point used by volume: %s", mountPoint));
        }
        return errors;
    }

    private Set<String> validateJobImage(String image) {
        HashSet errors = Sets.newHashSet();
        if (image == null) {
            errors.add("Image was not specified.");
        } else {
            this.validateImageReference(image, errors);
        }
        return errors;
    }

    private Set<String> validateJobId(Job job) {
        HashSet errors = Sets.newHashSet();
        JobId jobId = job.getId();
        if (jobId == null) {
            errors.add("Job id was not specified.");
            return errors;
        }
        String jobIdVersion = jobId.getVersion();
        String jobIdHash = jobId.getHash();
        JobId recomputedId = job.toBuilder().build().getId();
        errors.addAll(this.validateJobName(jobId, recomputedId));
        errors.addAll(this.validateJobVersion(jobIdVersion, recomputedId));
        if (this.shouldValidateJobHash) {
            errors.addAll(this.validateJobHash(jobIdHash, recomputedId));
        }
        return errors;
    }

    private Set<String> validateJobName(JobId jobId, JobId recomputedId) {
        HashSet errors = Sets.newHashSet();
        String jobIdName = jobId.getName();
        if (Strings.isNullOrEmpty((String)jobIdName)) {
            errors.add("Job name was not specified.");
            return errors;
        }
        if (!NAME_VERSION_PATTERN.matcher(jobIdName).matches()) {
            errors.add(String.format("Job name may only contain [0-9a-zA-Z-_.] in job name [%s].", recomputedId.getName()));
        }
        if (!recomputedId.getName().equals(jobIdName)) {
            errors.add(String.format("Id name mismatch: %s != %s", jobIdName, recomputedId.getName()));
        }
        return errors;
    }

    private Set<String> validateJobHostName(String hostname) {
        HashSet errors = Sets.newHashSet();
        if (Strings.isNullOrEmpty((String)hostname)) {
            return errors;
        }
        if (!HOSTNAME_PATTERN.matcher(hostname).matches()) {
            errors.add(String.format("Invalid hostname (%s), only [a-z0-9][a-z0-9-] are allowed, size between 1 and 63", hostname));
        }
        return errors;
    }

    private Set<String> validateJobVersion(String jobIdVersion, JobId recomputedId) {
        HashSet errors = Sets.newHashSet();
        if (Strings.isNullOrEmpty((String)jobIdVersion)) {
            errors.add(String.format("Job version was not specified in job id [%s].", recomputedId));
            return errors;
        }
        if (!NAME_VERSION_PATTERN.matcher(jobIdVersion).matches()) {
            errors.add(String.format("Job version may only contain [0-9a-zA-Z-_.] in job version [%s].", recomputedId.getVersion()));
        }
        if (!recomputedId.getVersion().equals(jobIdVersion)) {
            errors.add(String.format("Id version mismatch: %s != %s", jobIdVersion, recomputedId.getVersion()));
        }
        return errors;
    }

    private Set<String> validateJobHash(String jobIdHash, JobId recomputedId) {
        HashSet errors = Sets.newHashSet();
        if (Strings.isNullOrEmpty((String)jobIdHash)) {
            errors.add(String.format("Job hash was not specified in job id [%s].", recomputedId));
            return errors;
        }
        if (jobIdHash.indexOf(58) != -1) {
            errors.add(String.format("Job hash contains colon in job id [%s].", recomputedId));
        }
        if (!recomputedId.getHash().equals(jobIdHash)) {
            errors.add(String.format("Id hash mismatch: %s != %s", jobIdHash, recomputedId.getHash()));
        }
        return errors;
    }

    private boolean validateImageReference(String imageRef, Collection<String> errors) {
        String tag;
        String repo;
        boolean valid = true;
        int lastAtSign = imageRef.lastIndexOf(64);
        int lastColon = imageRef.lastIndexOf(58);
        if (lastAtSign != -1) {
            repo = imageRef.substring(0, lastAtSign);
            String digest = imageRef.substring(lastAtSign + 1);
            valid &= this.validateDigest(digest, errors);
        } else if (lastColon != -1 && !(tag = imageRef.substring(lastColon + 1)).contains("/")) {
            repo = imageRef.substring(0, lastColon);
            valid &= this.validateTag(tag, errors);
        } else {
            repo = imageRef;
        }
        String invalidRepoName = "Invalid repository name (ex: \"registry.domain.tld/myrepos\")";
        if (repo.contains("://")) {
            errors.add("Invalid repository name (ex: \"registry.domain.tld/myrepos\")");
            return false;
        }
        String[] nameParts = repo.split("/", 2);
        if (!(nameParts[0].contains(".") || nameParts[0].contains(":") || nameParts[0].equals("localhost"))) {
            return this.validateRepositoryName(imageRef, repo, errors);
        }
        if (nameParts.length < 2) {
            errors.add("Invalid repository name (ex: \"registry.domain.tld/myrepos\")");
            return false;
        }
        String endpoint = nameParts[0];
        String reposName = nameParts[1];
        valid &= this.validateEndpoint(endpoint, errors);
        return valid &= this.validateRepositoryName(imageRef, reposName, errors);
    }

    private boolean validateTag(String tag, Collection<String> errors) {
        if (tag.isEmpty()) {
            errors.add("Tag cannot be empty");
            return false;
        }
        if (!TAG_PATTERN.matcher(tag).matches()) {
            errors.add(String.format("Illegal tag: \"%s\", must match %s", tag, TAG_PATTERN));
            return false;
        }
        return true;
    }

    private boolean validateDigest(String digest, Collection<String> errors) {
        if (digest.isEmpty()) {
            errors.add("Digest cannot be empty");
            return false;
        }
        int firstColon = digest.indexOf(58);
        int lastColon = digest.lastIndexOf(58);
        if (firstColon <= 0 || firstColon != lastColon || firstColon == digest.length() - 1) {
            errors.add(String.format("Illegal digest: \"%s\"", digest));
            return false;
        }
        return true;
    }

    private boolean validateEndpoint(String endpoint, Collection<String> errors) {
        String[] parts = endpoint.split(":", 2);
        if (!this.validateAddress(parts[0], errors)) {
            return false;
        }
        if (parts.length > 1) {
            int port;
            try {
                port = Integer.valueOf(parts[1]);
            }
            catch (NumberFormatException e) {
                errors.add(String.format("Invalid port in endpoint: \"%s\"", endpoint));
                return false;
            }
            if (port < 0 || port > 65535) {
                errors.add(String.format("Invalid port in endpoint: \"%s\"", endpoint));
                return false;
            }
        }
        return true;
    }

    private boolean validateAddress(String address, Collection<String> errors) {
        if (IPV4_PATTERN.matcher(address).matches()) {
            return true;
        }
        if (!DOMAIN_PATTERN.matcher(address).matches() || DIGIT_PERIOD.matcher(address).find()) {
            errors.add(String.format("Invalid domain name: \"%s\"", address));
            return false;
        }
        return true;
    }

    private boolean validateRepositoryName(String imageName, String repositoryName, Collection<String> errors) {
        String[] nameParts;
        for (String name : nameParts = repositoryName.split("/")) {
            if (NAME_COMPONENT_PATTERN.matcher(name).matches()) continue;
            errors.add(String.format("Invalid image name (%s), only %s is allowed for each slash-separated name component (failed on \"%s\")", imageName, NAME_COMPONENT_PATTERN, name));
            return false;
        }
        if (repositoryName.length() > 255) {
            errors.add(String.format("Invalid image name (%s), repository name cannot be larger than %d characters", imageName, 255));
            return false;
        }
        return true;
    }

    private Set<String> validateJobHealthCheck(Job job) {
        HealthCheck healthCheck = job.getHealthCheck();
        if (healthCheck == null) {
            return Collections.emptySet();
        }
        HashSet errors = Sets.newHashSet();
        if (healthCheck instanceof ExecHealthCheck) {
            List<String> command = ((ExecHealthCheck)healthCheck).getCommand();
            if (command == null || command.isEmpty()) {
                errors.add("A command must be defined for `docker exec`-based health checks.");
            }
        } else if (healthCheck instanceof HttpHealthCheck || healthCheck instanceof TcpHealthCheck) {
            String port = healthCheck instanceof HttpHealthCheck ? ((HttpHealthCheck)healthCheck).getPort() : ((TcpHealthCheck)healthCheck).getPort();
            Map<String, PortMapping> ports = job.getPorts();
            if (Strings.isNullOrEmpty((String)port)) {
                errors.add("A port must be defined for HTTP and TCP health checks.");
            } else if (!ports.containsKey(port)) {
                errors.add(String.format("Health check port '%s' not defined in the job. Known ports are '%s'", port, Joiner.on((String)", ").join(ports.keySet())));
            }
        }
        return errors;
    }

    private Set<String> validateJobNetworkMode(Job job) {
        String networkMode = job.getNetworkMode();
        if (networkMode == null) {
            return Collections.emptySet();
        }
        HashSet errors = Sets.newHashSet();
        if (!VALID_NETWORK_MODES.contains(networkMode) && !networkMode.startsWith("container:")) {
            errors.add(String.format("A Docker container's network mode must be %s, or container:<name|id>.", Joiner.on((String)", ").join(VALID_NETWORK_MODES)));
        }
        return errors;
    }

    private Set<String> validateAddCapabilities(Job job) {
        Set<String> caps = job.getAddCapabilities();
        if (caps == null) {
            return Collections.emptySet();
        }
        HashSet errors = Sets.newHashSet();
        Sets.SetView disallowedCaps = Sets.difference(caps, this.whitelistedCapabilities);
        if (!disallowedCaps.isEmpty()) {
            errors.add(String.format("The following Linux capabilities aren't allowed by the Helios master: '%s'. The allowed capabilities are: '%s'.", Joiner.on((String)", ").join((Iterable)disallowedCaps), Joiner.on((String)", ").join(this.whitelistedCapabilities)));
        }
        return errors;
    }

    private boolean legalPort(int port) {
        return port >= 0 && port <= 65535;
    }
}

