/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.client.tls;

import com.eaio.uuid.UUID;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.BaseEncoding;
import com.spotify.helios.client.tls.CertificateAndPrivateKey;
import com.spotify.helios.client.tls.SshAgentContentSigner;
import com.spotify.helios.common.Hash;
import com.spotify.sshagentproxy.AgentProxy;
import com.spotify.sshagentproxy.Identity;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.MessageDigest;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.Calendar;
import java.util.Date;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.oiw.OIWObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.X500NameBuilder;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.SubjectKeyIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509ExtensionUtils;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.DigestCalculator;
import org.bouncycastle.operator.bc.BcDigestCalculatorProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class X509CertificateFactory {
    private static final Path HELIOS_HOME = Paths.get(System.getProperty("user.home"), ".helios");
    private static final BaseEncoding HEX_ENCODING = BaseEncoding.base16().lowerCase();
    private static final Logger log = LoggerFactory.getLogger(X509CertificateFactory.class);
    private static final JcaX509CertificateConverter CERTIFICATE_CONVERTER = new JcaX509CertificateConverter().setProvider("BC");
    private static final BaseEncoding KEY_ID_ENCODING = BaseEncoding.base16().upperCase().withSeparator(":", 2);
    private static final int KEY_SIZE = 2048;
    private final Path cacheDirectory;
    private final int validBeforeMilliseconds;
    private final int validAfterMilliseconds;

    public X509CertificateFactory() {
        this(HELIOS_HOME, (int)TimeUnit.HOURS.toMillis(1L), (int)TimeUnit.HOURS.toMillis(48L));
    }

    public X509CertificateFactory(Path cacheDirectory, int validBeforeMilliseconds, int validAfterMillieconds) {
        this.cacheDirectory = cacheDirectory;
        this.validBeforeMilliseconds = validBeforeMilliseconds;
        this.validAfterMilliseconds = validAfterMillieconds;
    }

    public CertificateAndPrivateKey get(AgentProxy agentProxy, Identity identity, String username) {
        X509Certificate cachedX509;
        Date now;
        MessageDigest identityHash = Hash.sha1();
        identityHash.update(identity.getKeyBlob());
        identityHash.update(username.getBytes());
        String identityHex = HEX_ENCODING.encode(identityHash.digest()).substring(0, 8);
        Path cacheCertPath = this.cacheDirectory.resolve(identityHex + ".crt");
        Path cacheKeyPath = this.cacheDirectory.resolve(identityHex + ".pem");
        boolean useCached = false;
        CertificateAndPrivateKey cached = null;
        try {
            if (Files.exists(cacheCertPath, new LinkOption[0]) && Files.exists(cacheKeyPath, new LinkOption[0])) {
                cached = CertificateAndPrivateKey.from(cacheCertPath, cacheKeyPath);
            }
        }
        catch (IOException | GeneralSecurityException e) {
            log.debug("error reading cached certificate and key from {} for identity={}", new Object[]{this.cacheDirectory, identity.getComment(), e});
        }
        if (cached != null && cached.getCertificate() instanceof X509Certificate && (now = new Date()).after((cachedX509 = (X509Certificate)cached.getCertificate()).getNotBefore()) && now.before(cachedX509.getNotAfter())) {
            useCached = true;
        }
        if (useCached) {
            log.debug("using existing certificate for {} from {}", (Object)username, (Object)cacheCertPath);
            return cached;
        }
        CertificateAndPrivateKey generated = this.generate(agentProxy, identity, username);
        X509CertificateFactory.saveToCache(this.cacheDirectory, cacheCertPath, cacheKeyPath, generated);
        return generated;
    }

    private CertificateAndPrivateKey generate(AgentProxy agentProxy, Identity identity, String username) {
        UUID uuid = new UUID();
        Calendar calendar = Calendar.getInstance();
        X500Name issuerDN = new X500Name("C=US,O=Spotify,CN=helios-client");
        X500Name subjectDN = new X500NameBuilder().addRDN(BCStyle.UID, username).build();
        calendar.add(14, -this.validBeforeMilliseconds);
        Date notBefore = calendar.getTime();
        calendar.add(14, this.validBeforeMilliseconds + this.validAfterMilliseconds);
        Date notAfter = calendar.getTime();
        BigInteger serialNumber = BigInteger.valueOf(uuid.getTime()).abs();
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA", "BC");
            keyPairGenerator.initialize(2048, new SecureRandom());
            KeyPair keyPair = keyPairGenerator.generateKeyPair();
            SubjectPublicKeyInfo subjectPublicKeyInfo = SubjectPublicKeyInfo.getInstance((Object)ASN1Sequence.getInstance((Object)keyPair.getPublic().getEncoded()));
            X509v3CertificateBuilder builder = new X509v3CertificateBuilder(issuerDN, serialNumber, notBefore, notAfter, subjectDN, subjectPublicKeyInfo);
            DigestCalculator digestCalculator = new BcDigestCalculatorProvider().get(new AlgorithmIdentifier(OIWObjectIdentifiers.idSHA1));
            X509ExtensionUtils utils = new X509ExtensionUtils(digestCalculator);
            SubjectKeyIdentifier keyId = utils.createSubjectKeyIdentifier(subjectPublicKeyInfo);
            String keyIdHex = KEY_ID_ENCODING.encode(keyId.getKeyIdentifier());
            log.info("generating an X509 certificate for {} with key ID={} and identity={}", new Object[]{username, keyIdHex, identity.getComment()});
            builder.addExtension(Extension.subjectKeyIdentifier, false, (ASN1Encodable)keyId);
            builder.addExtension(Extension.authorityKeyIdentifier, false, (ASN1Encodable)utils.createAuthorityKeyIdentifier(subjectPublicKeyInfo));
            builder.addExtension(Extension.keyUsage, false, (ASN1Encodable)new KeyUsage(132));
            builder.addExtension(Extension.basicConstraints, true, (ASN1Encodable)new BasicConstraints(false));
            X509CertificateHolder holder = builder.build((ContentSigner)new SshAgentContentSigner(agentProxy, identity));
            X509Certificate certificate = CERTIFICATE_CONVERTER.getCertificate(holder);
            log.debug("generated certificate:\n{}", (Object)X509CertificateFactory.asPEMString(certificate));
            return new CertificateAndPrivateKey(certificate, keyPair.getPrivate());
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static void saveToCache(Path cacheDirectory, Path cacheCertPath, Path cacheKeyPath, CertificateAndPrivateKey certificateAndPrivateKey) {
        try {
            Files.createDirectories(cacheDirectory, new FileAttribute[0]);
            String certPem = X509CertificateFactory.asPEMString(certificateAndPrivateKey.getCertificate());
            String keyPem = X509CertificateFactory.asPEMString(certificateAndPrivateKey.getPrivateKey());
            ImmutableSet options = ImmutableSet.of((Object)StandardOpenOption.CREATE, (Object)StandardOpenOption.WRITE);
            ImmutableSet perms = ImmutableSet.of((Object)((Object)PosixFilePermission.OWNER_READ), (Object)((Object)PosixFilePermission.OWNER_WRITE));
            FileAttribute<Set<PosixFilePermission>> attrs = PosixFilePermissions.asFileAttribute((Set<PosixFilePermission>)perms);
            try (SeekableByteChannel sbc = Files.newByteChannel(cacheCertPath, (Set<? extends OpenOption>)options, attrs);){
                sbc.write(ByteBuffer.wrap(certPem.getBytes()));
            }
            sbc = Files.newByteChannel(cacheKeyPath, (Set<? extends OpenOption>)options, attrs);
            var10_11 = null;
            try {
                sbc.write(ByteBuffer.wrap(keyPem.getBytes()));
            }
            catch (Throwable throwable) {
                var10_11 = throwable;
                throw throwable;
            }
            finally {
                if (sbc != null) {
                    if (var10_11 != null) {
                        try {
                            sbc.close();
                        }
                        catch (Throwable throwable) {
                            var10_11.addSuppressed(throwable);
                        }
                    } else {
                        sbc.close();
                    }
                }
            }
            log.debug("cached generated certificate to {}", (Object)cacheCertPath);
        }
        catch (IOException e) {
            log.warn("error caching generated certificate", (Throwable)e);
        }
    }

    private static String asPEMString(Object o) throws IOException {
        StringWriter sw = new StringWriter();
        try (JcaPEMWriter pw = new JcaPEMWriter((Writer)sw);){
            pw.writeObject(o);
        }
        return sw.toString();
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }
}

