/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.cli.command;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.spotify.helios.cli.command.WildcardJobCommand;
import com.spotify.helios.client.HeliosClient;
import com.spotify.helios.common.Json;
import com.spotify.helios.common.descriptors.JobId;
import com.spotify.helios.common.descriptors.RolloutOptions;
import com.spotify.helios.common.descriptors.TaskStatus;
import com.spotify.helios.common.protocol.DeploymentGroupStatusResponse;
import com.spotify.helios.common.protocol.RollingUpdateResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import net.sourceforge.argparse4j.impl.Arguments;
import net.sourceforge.argparse4j.inf.Argument;
import net.sourceforge.argparse4j.inf.ArgumentAction;
import net.sourceforge.argparse4j.inf.Namespace;
import net.sourceforge.argparse4j.inf.Subparser;

public class RollingUpdateCommand
extends WildcardJobCommand {
    private static final long POLL_INTERVAL_MILLIS = 1000L;
    private final SleepFunction sleepFunction;
    private final Supplier<Long> timeSupplier;
    private final Argument nameArg;
    private final Argument timeoutArg;
    private final Argument parallelismArg;
    private final Argument asyncArg;
    private final Argument rolloutTimeoutArg;
    private final Argument migrateArg;
    private final Argument overlapArg;
    private final Argument tokenArg;
    private final Argument ignoreFailuresArg;

    public RollingUpdateCommand(Subparser parser) {
        this(parser, new SleepFunction(){

            @Override
            public void sleep(long millis) throws InterruptedException {
                Thread.sleep(millis);
            }
        }, new Supplier<Long>(){

            public Long get() {
                return System.currentTimeMillis();
            }
        });
    }

    @VisibleForTesting
    RollingUpdateCommand(Subparser parser, SleepFunction sleepFunction, Supplier<Long> timeSupplier) {
        super(parser, true);
        this.sleepFunction = sleepFunction;
        this.timeSupplier = timeSupplier;
        parser.help("Initiate a rolling update");
        this.nameArg = parser.addArgument(new String[]{"deployment-group-name"}).required(true).help("Deployment group name");
        this.timeoutArg = parser.addArgument(new String[]{"-t", "--timeout"}).setDefault((Object)RolloutOptions.DEFAULT_TIMEOUT).type(Long.class).help("Fail rollout if a job takes longer than this to reach RUNNING (seconds)");
        this.parallelismArg = parser.addArgument(new String[]{"-p", "--par"}).dest("parallelism").setDefault((Object)1).type(Integer.class).help("Number of hosts to deploy to concurrently");
        this.asyncArg = parser.addArgument(new String[]{"--async"}).action((ArgumentAction)Arguments.storeTrue()).help("Don't block until rolling-update is complete");
        this.rolloutTimeoutArg = parser.addArgument(new String[]{"-T", "--rollout-timeout"}).setDefault((Object)60L).type(Long.class).help("Exit if rolling-update takes longer than the given value (minutes). Note that this will NOT abort the rolling update, it will just cause this command to exit.");
        this.migrateArg = parser.addArgument(new String[]{"--migrate"}).setDefault((Object)false).action((ArgumentAction)Arguments.storeTrue()).help("When specified a rolling-update will undeploy not only jobs previously deployed by the deployment-group but also jobs with the same job id. Use it ONCE when migrating a service to using deployment-groups");
        this.overlapArg = parser.addArgument(new String[]{"--overlap"}).setDefault((Object)false).action((ArgumentAction)Arguments.storeTrue()).help("When specified a rolling-update will, for every host, first deploy the new version of a job before undeploying the old one. Note that the command will fail if the job contains static port assignments.");
        this.tokenArg = parser.addArgument(new String[]{"--token"}).nargs("?").setDefault((Object)"").help("Insecure access token meant to prevent accidental changes to your job (e.g. undeploys).");
        this.ignoreFailuresArg = parser.addArgument(new String[]{"--ignore-failures"}).setDefault((Object)false).action((ArgumentAction)Arguments.storeTrue()).help("When specified, the rolling-update will ignore *all* failures and will proceed to deploying the job to all hosts in the deployment group. The rolling-update will go through the normal rollout plan (respecting the --par and --overlap settings), and will wait for the job to reach RUNNING on each host as normal; however, any failure that would otherwise cause the rolling-update to abort and set the deployment group's status to FAILED is *ignored*. Be *VERY* careful about using this option, as it has the potential to completely take down your service by rolling out a broken job to all of the hosts in your group.");
    }

    @Override
    protected int runWithJobId(Namespace options, HeliosClient client, PrintStream out, boolean json, JobId jobId, BufferedReader stdin) throws ExecutionException, InterruptedException, IOException {
        String name = options.getString(this.nameArg.getDest());
        long timeout = options.getLong(this.timeoutArg.getDest());
        int parallelism = options.getInt(this.parallelismArg.getDest());
        boolean async = options.getBoolean(this.asyncArg.getDest());
        long rolloutTimeout = options.getLong(this.rolloutTimeoutArg.getDest());
        boolean migrate = options.getBoolean(this.migrateArg.getDest());
        boolean overlap = options.getBoolean(this.overlapArg.getDest());
        String token = options.getString(this.tokenArg.getDest());
        boolean ignoreFailures = options.getBoolean(this.ignoreFailuresArg.getDest());
        Preconditions.checkArgument((timeout > 0L ? 1 : 0) != 0, (Object)"Timeout must be greater than 0");
        Preconditions.checkArgument((parallelism > 0 ? 1 : 0) != 0, (Object)"Parallelism must be greater than 0");
        Preconditions.checkArgument((rolloutTimeout > 0L ? 1 : 0) != 0, (Object)"Rollout timeout must be greater than 0");
        long startTime = (Long)this.timeSupplier.get();
        RolloutOptions rolloutOptions = RolloutOptions.newBuilder().setTimeout(timeout).setParallelism(parallelism).setMigrate(migrate).setOverlap(overlap).setToken(token).setIgnoreFailures(ignoreFailures).build();
        RollingUpdateResponse response = (RollingUpdateResponse)client.rollingUpdate(name, jobId, rolloutOptions).get();
        if (response.getStatus() != RollingUpdateResponse.Status.OK) {
            if (!json) {
                out.println("Failed: " + response);
            } else {
                out.println(response.toJsonString());
            }
            return 1;
        }
        if (!json) {
            out.println(String.format("Rolling update%s started: %s -> %s (parallelism=%d, timeout=%d, overlap=%b, token=%s, ignoreFailures=%b)%s", async ? " (async)" : "", name, jobId.toShortString(), parallelism, timeout, overlap, token, ignoreFailures, async ? "" : "\n"));
        }
        HashMap jsonOutput = Maps.newHashMap();
        jsonOutput.put("parallelism", parallelism);
        jsonOutput.put("timeout", timeout);
        jsonOutput.put("overlap", overlap);
        jsonOutput.put("token", token);
        jsonOutput.put("ignoreFailures", ignoreFailures);
        if (async) {
            if (json) {
                jsonOutput.put("status", response.getStatus());
                out.println(Json.asStringUnchecked((Object)jsonOutput));
            }
            return 0;
        }
        String error = "";
        boolean failed = false;
        boolean timedOut = false;
        HashSet reported = Sets.newHashSet();
        while (true) {
            DeploymentGroupStatusResponse status;
            if ((status = (DeploymentGroupStatusResponse)client.deploymentGroupStatus(name).get()) == null) {
                failed = true;
                error = "Failed to fetch deployment-group status";
                break;
            }
            if (!jobId.equals((Object)status.getDeploymentGroup().getJobId())) {
                failed = true;
                error = "Deployment-group job id changed during rolling-update";
                break;
            }
            if (!json) {
                for (DeploymentGroupStatusResponse.HostStatus hostStatus : status.getHostStatuses()) {
                    JobId hostJobId = hostStatus.getJobId();
                    String host = hostStatus.getHost();
                    TaskStatus.State state = hostStatus.getState();
                    boolean done = hostJobId != null && hostJobId.equals((Object)jobId) && state == TaskStatus.State.RUNNING;
                    if (!done || !reported.add(host)) continue;
                    out.println(String.format("%s -> %s (%d/%d)", host, state, reported.size(), status.getHostStatuses().size()));
                }
            }
            if (status.getStatus() != DeploymentGroupStatusResponse.Status.ROLLING_OUT) {
                if (status.getStatus() != DeploymentGroupStatusResponse.Status.FAILED) break;
                failed = true;
                error = status.getError();
                break;
            }
            if ((Long)this.timeSupplier.get() - startTime > TimeUnit.MINUTES.toMillis(rolloutTimeout)) {
                timedOut = true;
                break;
            }
            this.sleepFunction.sleep(1000L);
        }
        double duration = (double)((Long)this.timeSupplier.get() - startTime) / 1000.0;
        if (json) {
            if (failed) {
                jsonOutput.put("status", "FAILED");
                jsonOutput.put("error", error);
            } else if (timedOut) {
                jsonOutput.put("status", "TIMEOUT");
            } else {
                jsonOutput.put("status", "DONE");
            }
            jsonOutput.put("duration", duration);
            out.println(Json.asStringUnchecked((Object)jsonOutput));
        } else {
            out.println();
            if (failed) {
                out.println(String.format("Failed: %s", error));
            } else if (timedOut) {
                out.println("Timed out! (rolling-update still in progress)");
            } else {
                out.println("Done.");
            }
            out.println(String.format("Duration: %.2f s", duration));
        }
        return failed || timedOut ? 1 : 0;
    }

    static interface SleepFunction {
        public void sleep(long var1) throws InterruptedException;
    }
}

