/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.AuthProvider;
import com.datastax.driver.core.Authenticator;
import com.datastax.driver.core.BusyConnectionException;
import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ClusterNameMismatchException;
import com.datastax.driver.core.Configuration;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.DefaultResultSetFuture;
import com.datastax.driver.core.ExceptionCode;
import com.datastax.driver.core.ExecutionInfo;
import com.datastax.driver.core.Frame;
import com.datastax.driver.core.FrameCompressor;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostConnectionPool;
import com.datastax.driver.core.Message;
import com.datastax.driver.core.NettyOptions;
import com.datastax.driver.core.OperationTimedOutException;
import com.datastax.driver.core.ProtocolOptions;
import com.datastax.driver.core.ProtocolV1Authenticator;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.RequestHandler;
import com.datastax.driver.core.Requests;
import com.datastax.driver.core.Responses;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.SSLOptions;
import com.datastax.driver.core.SocketOptions;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.StreamIdGenerator;
import com.datastax.driver.core.SystemProperties;
import com.datastax.driver.core.TransportException;
import com.datastax.driver.core.UnsupportedProtocolVersionException;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.exceptions.DriverException;
import com.datastax.driver.core.exceptions.DriverInternalError;
import com.datastax.driver.core.utils.MoreFutures;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.MapMaker;
import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.Uninterruptibles;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.ChannelOption;
import io.netty.channel.ChannelPipeline;
import io.netty.channel.EventLoop;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.channel.group.ChannelGroup;
import io.netty.channel.group.DefaultChannelGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.Timeout;
import io.netty.util.Timer;
import io.netty.util.TimerTask;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.GlobalEventExecutor;
import java.lang.ref.WeakReference;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.ClosedChannelException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import javax.net.ssl.SSLEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Connection {
    private static final Logger logger = LoggerFactory.getLogger(Connection.class);
    private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
    private static final boolean DISABLE_COALESCING = SystemProperties.getBoolean((String)"com.datastax.driver.DISABLE_COALESCING", (boolean)false);
    final AtomicReference<State> state = new AtomicReference<State>(State.OPEN);
    volatile long maxIdleTime;
    public final InetSocketAddress address;
    private final String name;
    @VisibleForTesting
    volatile Channel channel;
    private final Factory factory;
    @VisibleForTesting
    final Dispatcher dispatcher;
    public final AtomicInteger inFlight = new AtomicInteger(0);
    private final AtomicInteger writer = new AtomicInteger(0);
    private volatile String keyspace;
    private volatile boolean isInitialized;
    private final AtomicBoolean isDefunct = new AtomicBoolean();
    private final AtomicBoolean signaled = new AtomicBoolean();
    private final AtomicReference<ConnectionCloseFuture> closeFuture = new AtomicReference();
    private final AtomicReference<Owner> ownerRef = new AtomicReference();
    private static final ConcurrentMap<EventLoop, Flusher> flusherLookup = new MapMaker().concurrencyLevel(16).weakKeys().makeMap();
    private static final ResponseCallback HEARTBEAT_CALLBACK = new ResponseCallback(){

        @Override
        public Message.Request request() {
            return new Requests.Options();
        }

        @Override
        public int retryCount() {
            return 0;
        }

        @Override
        public void onSet(Connection connection, Message.Response response, long l, int n) {
            switch (response.type) {
                case SUPPORTED: {
                    logger.debug("{} heartbeat query succeeded", (Object)connection);
                    break;
                }
                default: {
                    this.fail(connection, (Exception)((Object)new ConnectionException(connection.address, "Unexpected heartbeat response: " + response)));
                }
            }
        }

        @Override
        public void onException(Connection connection, Exception exception, long l, int n) {
        }

        @Override
        public boolean onTimeout(Connection connection, long l, int n) {
            this.fail(connection, (Exception)((Object)new ConnectionException(connection.address, "Heartbeat query timed out")));
            return true;
        }

        private void fail(Connection connection, Exception exception) {
            connection.defunct(exception);
        }
    };

    protected Connection(String string, InetSocketAddress inetSocketAddress, Factory factory, Owner owner) {
        this.address = inetSocketAddress;
        this.factory = factory;
        this.dispatcher = new Dispatcher();
        this.name = string;
        this.ownerRef.set(owner);
    }

    Connection(String string, InetSocketAddress inetSocketAddress, Factory factory) {
        this(string, inetSocketAddress, factory, null);
    }

    public ListenableFuture<Void> initAsync() {
        ChannelFuture channelFuture;
        ProtocolOptions protocolOptions;
        Object object;
        if (this.factory.isShutdown) {
            return Futures.immediateFailedFuture((Throwable)new ConnectionException(this.address, "Connection factory is shut down"));
        }
        ProtocolVersion protocolVersion = this.factory.protocolVersion == null ? ProtocolVersion.NEWEST_SUPPORTED : this.factory.protocolVersion;
        final SettableFuture settableFuture = SettableFuture.create();
        try {
            object = this.factory.newBootstrap();
            protocolOptions = this.factory.configuration.getProtocolOptions();
            object.handler((ChannelHandler)new Initializer(this, protocolVersion, protocolOptions.getCompression().compressor(), protocolOptions.getSSLOptions(), this.factory.configuration.getPoolingOptions().getHeartbeatIntervalSeconds(), this.factory.configuration.getNettyOptions()));
            channelFuture = object.connect((SocketAddress)this.address);
            this.writer.incrementAndGet();
            channelFuture.addListener((GenericFutureListener)new ChannelFutureListener(){

                public void operationComplete(ChannelFuture channelFuture) throws Exception {
                    Connection.this.writer.decrementAndGet();
                    Connection.this.channel = channelFuture.channel();
                    if (Connection.this.isClosed()) {
                        Connection.this.channel.close().addListener((GenericFutureListener)new ChannelFutureListener(){

                            public void operationComplete(ChannelFuture channelFuture) throws Exception {
                                settableFuture.setException((Throwable)new TransportException(Connection.this.address, "Connection closed during initialization."));
                            }
                        });
                    } else {
                        Connection.this.factory.allChannels.add((Object)Connection.this.channel);
                        if (!channelFuture.isSuccess()) {
                            if (logger.isDebugEnabled()) {
                                logger.debug(String.format("%s Error connecting to %s%s", Connection.this, Connection.this.address, Connection.extractMessage(channelFuture.cause())));
                            }
                            settableFuture.setException((Throwable)new TransportException(Connection.this.address, "Cannot connect", channelFuture.cause()));
                        } else {
                            logger.debug("{} Connection established, initializing transport", (Object)Connection.this);
                            Connection.this.channel.closeFuture().addListener((GenericFutureListener)new ChannelCloseListener());
                            settableFuture.set(null);
                        }
                    }
                }
            });
        }
        catch (RuntimeException runtimeException) {
            this.closeAsync().force();
            throw runtimeException;
        }
        object = this.factory.manager.configuration.getPoolingOptions().getInitializationExecutor();
        protocolOptions = Futures.transformAsync((ListenableFuture)settableFuture, this.onChannelReady(protocolVersion, (Executor)object), (Executor)object);
        channelFuture = Futures.catchingAsync((ListenableFuture)protocolOptions, Throwable.class, (AsyncFunction)new AsyncFunction<Throwable, Void>(){

            public ListenableFuture<Void> apply(@Nullable Throwable throwable) throws Exception {
                SettableFuture settableFuture = SettableFuture.create();
                if (throwable instanceof ClusterNameMismatchException || throwable instanceof UnsupportedProtocolVersionException) {
                    Connection.this.closeAsync().force();
                    settableFuture.setException(throwable);
                } else {
                    Exception exception = throwable instanceof ConnectionException || throwable instanceof DriverException || throwable instanceof InterruptedException ? (Exception)throwable : new ConnectionException(Connection.this.address, String.format("Unexpected error during transport initialization (%s)", throwable), throwable);
                    settableFuture.setException((Throwable)Connection.this.defunct(exception));
                }
                return settableFuture;
            }
        }, (Executor)object);
        Futures.addCallback((ListenableFuture)channelFuture, (FutureCallback)new MoreFutures.FailureCallback<Void>(){

            public void onFailure(Throwable throwable) {
                if (!Connection.this.isClosed()) {
                    Connection.this.closeAsync().force();
                }
            }
        }, (Executor)object);
        return channelFuture;
    }

    private static String extractMessage(Throwable throwable) {
        if (throwable == null) {
            return "";
        }
        String string = throwable.getMessage() == null || throwable.getMessage().isEmpty() ? throwable.toString() : throwable.getMessage();
        return " (" + string + ')';
    }

    private AsyncFunction<Void, Void> onChannelReady(final ProtocolVersion protocolVersion, final Executor executor) {
        return new AsyncFunction<Void, Void>(){

            public ListenableFuture<Void> apply(Void void_) throws Exception {
                ProtocolOptions.Compression compression = ((Connection)Connection.this).factory.configuration.getProtocolOptions().getCompression();
                Future future = Connection.this.write((Message.Request)new Requests.Startup(compression));
                return Futures.transformAsync((ListenableFuture)future, (AsyncFunction)Connection.this.onStartupResponse(protocolVersion, executor), (Executor)executor);
            }
        };
    }

    private AsyncFunction<Message.Response, Void> onStartupResponse(final ProtocolVersion protocolVersion, final Executor executor) {
        return new AsyncFunction<Message.Response, Void>(){

            public ListenableFuture<Void> apply(Message.Response response) throws Exception {
                switch (response.type) {
                    case READY: {
                        return Connection.this.checkClusterName(protocolVersion, executor);
                    }
                    case ERROR: {
                        Responses.Error error = (Responses.Error)response;
                        if (error.code == ExceptionCode.PROTOCOL_ERROR && error.message.contains("Invalid or unsupported protocol version")) {
                            throw Connection.this.unsupportedProtocolVersionException(protocolVersion, error.serverProtocolVersion);
                        }
                        throw new TransportException(Connection.this.address, String.format("Error initializing connection: %s", error.message));
                    }
                    case AUTHENTICATE: {
                        Authenticator authenticator = ((Connection)Connection.this).factory.authProvider.newAuthenticator(Connection.this.address);
                        switch (protocolVersion) {
                            case V1: {
                                if (authenticator instanceof ProtocolV1Authenticator) {
                                    return Connection.this.authenticateV1(authenticator, protocolVersion, executor);
                                }
                                return Connection.this.authenticateV2(authenticator, protocolVersion, executor);
                            }
                            case V2: 
                            case V3: {
                                return Connection.this.authenticateV2(authenticator, protocolVersion, executor);
                            }
                        }
                        throw Connection.this.defunct(protocolVersion.unsupported());
                    }
                }
                throw new TransportException(Connection.this.address, String.format("Unexpected %s response message from server to a STARTUP message", response.type));
            }
        };
    }

    private ListenableFuture<Void> checkClusterName(ProtocolVersion protocolVersion, Executor executor) {
        final String string = this.factory.manager.metadata.clusterName;
        if (string == null) {
            this.markInitialized();
            return MoreFutures.VOID_SUCCESS;
        }
        DefaultResultSetFuture defaultResultSetFuture = new DefaultResultSetFuture(null, protocolVersion, (Message.Request)new Requests.Query("select cluster_name from system.local"));
        try {
            this.write((ResponseCallback)defaultResultSetFuture);
            return Futures.transformAsync((ListenableFuture)defaultResultSetFuture, (AsyncFunction)new AsyncFunction<ResultSet, Void>(){

                public ListenableFuture<Void> apply(ResultSet resultSet) throws Exception {
                    Row row = resultSet.one();
                    String string2 = row.getString("cluster_name");
                    if (!string.equals(string2)) {
                        throw new ClusterNameMismatchException(Connection.this.address, string2, string);
                    }
                    Connection.this.markInitialized();
                    return MoreFutures.VOID_SUCCESS;
                }
            }, (Executor)executor);
        }
        catch (Exception exception) {
            return Futures.immediateFailedFuture((Throwable)exception);
        }
    }

    private void markInitialized() {
        this.isInitialized = true;
        Host.statesLogger.debug("[{}] {} Transport initialized, connection ready", (Object)this.address, (Object)this);
    }

    private ListenableFuture<Void> authenticateV1(Authenticator authenticator, final ProtocolVersion protocolVersion, final Executor executor) {
        Requests.Credentials credentials = new Requests.Credentials(((ProtocolV1Authenticator)authenticator).getCredentials());
        try {
            Future future = this.write((Message.Request)credentials);
            return Futures.transformAsync((ListenableFuture)future, (AsyncFunction)new AsyncFunction<Message.Response, Void>(){

                public ListenableFuture<Void> apply(Message.Response response) throws Exception {
                    switch (response.type) {
                        case READY: {
                            return Connection.this.checkClusterName(protocolVersion, executor);
                        }
                        case ERROR: {
                            throw new AuthenticationException(Connection.this.address, ((Responses.Error)response).message);
                        }
                    }
                    throw new TransportException(Connection.this.address, String.format("Unexpected %s response message from server to a CREDENTIALS message", response.type));
                }
            }, (Executor)executor);
        }
        catch (Exception exception) {
            return Futures.immediateFailedFuture((Throwable)exception);
        }
    }

    private ListenableFuture<Void> authenticateV2(Authenticator authenticator, ProtocolVersion protocolVersion, Executor executor) {
        byte[] byArray = authenticator.initialResponse();
        if (null == byArray) {
            byArray = EMPTY_BYTE_ARRAY;
        }
        try {
            Future future = this.write((Message.Request)new Requests.AuthResponse(byArray));
            return Futures.transformAsync((ListenableFuture)future, this.onV2AuthResponse(authenticator, protocolVersion, executor), (Executor)executor);
        }
        catch (Exception exception) {
            return Futures.immediateFailedFuture((Throwable)exception);
        }
    }

    private AsyncFunction<Message.Response, Void> onV2AuthResponse(final Authenticator authenticator, final ProtocolVersion protocolVersion, final Executor executor) {
        return new AsyncFunction<Message.Response, Void>(){

            public ListenableFuture<Void> apply(Message.Response response) throws Exception {
                switch (response.type) {
                    case AUTH_SUCCESS: {
                        logger.trace("{} Authentication complete", (Object)this);
                        authenticator.onAuthenticationSuccess(((Responses.AuthSuccess)response).token);
                        return Connection.this.checkClusterName(protocolVersion, executor);
                    }
                    case AUTH_CHALLENGE: {
                        byte[] byArray = authenticator.evaluateChallenge(((Responses.AuthChallenge)response).token);
                        if (byArray == null) {
                            logger.trace("{} Authentication complete (No response to server)", (Object)this);
                            return Connection.this.checkClusterName(protocolVersion, executor);
                        }
                        logger.trace("{} Sending Auth response to challenge", (Object)this);
                        Future future = Connection.this.write((Message.Request)new Requests.AuthResponse(byArray));
                        return Futures.transformAsync((ListenableFuture)future, (AsyncFunction)Connection.this.onV2AuthResponse(authenticator, protocolVersion, executor), (Executor)executor);
                    }
                    case ERROR: {
                        String string = ((Responses.Error)response).message;
                        if (string.startsWith("java.lang.ArrayIndexOutOfBoundsException: 15")) {
                            string = String.format("Cannot use authenticator %s with protocol version 1, only plain text authentication is supported with this protocol version", authenticator);
                        }
                        throw new AuthenticationException(Connection.this.address, string);
                    }
                }
                throw new TransportException(Connection.this.address, String.format("Unexpected %s response message from server to authentication message", response.type));
            }
        };
    }

    private UnsupportedProtocolVersionException unsupportedProtocolVersionException(ProtocolVersion protocolVersion, ProtocolVersion protocolVersion2) {
        logger.debug("Got unsupported protocol version error from {} for version {} server supports version {}", new Object[]{this.address, protocolVersion, protocolVersion2});
        return new UnsupportedProtocolVersionException(this.address, protocolVersion, protocolVersion2);
    }

    public boolean isDefunct() {
        return this.isDefunct.get();
    }

    public int maxAvailableStreams() {
        return this.dispatcher.streamIdHandler.maxAvailableStreams();
    }

    <E extends Exception> E defunct(E e) {
        if (this.isDefunct.compareAndSet(false, true)) {
            if (Host.statesLogger.isTraceEnabled()) {
                Host.statesLogger.trace("Defuncting " + this, e);
            } else if (Host.statesLogger.isDebugEnabled()) {
                Host.statesLogger.debug("Defuncting {} because: {}", (Object)this, (Object)e.getMessage());
            }
            Host host = this.factory.manager.metadata.getHost(this.address);
            if (host != null) {
                boolean bl = this.signaled.compareAndSet(false, true);
                boolean bl2 = host.convictionPolicy.signalConnectionFailure(this, bl);
                if (bl2) {
                    this.factory.manager.signalHostDown(host, host.wasJustAdded());
                } else {
                    this.notifyOwnerWhenDefunct();
                }
            }
            this.closeAsync().force();
        }
        return e;
    }

    private void notifyOwnerWhenDefunct() {
        if (!this.isInitialized) {
            return;
        }
        Owner owner = this.ownerRef.get();
        if (owner != null) {
            owner.onConnectionDefunct(this);
        }
    }

    public String keyspace() {
        return this.keyspace;
    }

    public void setKeyspace(String string) throws ConnectionException {
        if (string == null) {
            return;
        }
        if (this.keyspace != null && this.keyspace.equals(string)) {
            return;
        }
        try {
            Uninterruptibles.getUninterruptibly(this.setKeyspaceAsync(string));
        }
        catch (ConnectionException connectionException) {
            throw this.defunct(connectionException);
        }
        catch (BusyConnectionException busyConnectionException) {
            logger.warn("Tried to set the keyspace on busy {}. This should not happen but is not critical (it will be retried)", (Object)this);
            throw new ConnectionException(this.address, "Tried to set the keyspace on busy connection");
        }
        catch (ExecutionException executionException) {
            Throwable throwable = executionException.getCause();
            if (throwable instanceof OperationTimedOutException) {
                logger.warn("Timeout while setting keyspace on {}. This should not happen but is not critical (it will be retried)", (Object)this);
                throw new ConnectionException(this.address, "Timeout while setting keyspace on connection");
            }
            throw this.defunct(new ConnectionException(this.address, "Error while setting keyspace", throwable));
        }
    }

    ListenableFuture<Void> setKeyspaceAsync(String string) throws ConnectionException, BusyConnectionException {
        logger.trace("{} Setting keyspace {}", (Object)this, (Object)string);
        Future future = this.write((Message.Request)new Requests.Query("USE \"" + string + '\"'));
        return Futures.transformAsync((ListenableFuture)future, (AsyncFunction)new AsyncFunction<Message.Response, Void>(){

            public ListenableFuture<Void> apply(Message.Response response) throws Exception {
                if (response instanceof Responses.Result.SetKeyspace) {
                    Connection.this.keyspace = ((Responses.Result.SetKeyspace)response).keyspace;
                    return MoreFutures.VOID_SUCCESS;
                }
                if (response.type == Message.Response.Type.ERROR) {
                    Responses.Error error = (Responses.Error)response;
                    throw Connection.this.defunct(error.asException(Connection.this.address));
                }
                throw Connection.this.defunct(new DriverInternalError("Unexpected response while setting keyspace: " + response));
            }
        }, (Executor)this.factory.manager.configuration.getPoolingOptions().getInitializationExecutor());
    }

    public Future write(Message.Request request) throws ConnectionException, BusyConnectionException {
        Future future = new Future(request);
        this.write((ResponseCallback)((Object)future));
        return future;
    }

    public ResponseHandler write(ResponseCallback responseCallback) throws ConnectionException, BusyConnectionException {
        return this.write(responseCallback, true);
    }

    public ResponseHandler write(ResponseCallback responseCallback, boolean bl) throws ConnectionException, BusyConnectionException {
        ResponseHandler responseHandler = new ResponseHandler(this, responseCallback);
        this.dispatcher.add(responseHandler);
        Message.Request request = responseCallback.request().setStreamId(responseHandler.streamId);
        if (this.isDefunct.get()) {
            this.dispatcher.removeHandler(responseHandler, true);
            throw new ConnectionException(this.address, "Write attempt on defunct connection");
        }
        if (this.isClosed()) {
            this.dispatcher.removeHandler(responseHandler, true);
            throw new ConnectionException(this.address, "Connection has been closed");
        }
        logger.trace("{}, stream {}, writing request {}", new Object[]{this, request.getStreamId(), request});
        this.writer.incrementAndGet();
        if (DISABLE_COALESCING) {
            this.channel.writeAndFlush((Object)request).addListener((GenericFutureListener)this.writeHandler(request, responseHandler));
        } else {
            this.flush(new FlushItem(this.channel, request, this.writeHandler(request, responseHandler)));
        }
        if (bl) {
            responseHandler.startTimeout();
        }
        return responseHandler;
    }

    private ChannelFutureListener writeHandler(final Message.Request request, final ResponseHandler responseHandler) {
        return new ChannelFutureListener(){

            public void operationComplete(ChannelFuture channelFuture) {
                Connection.this.writer.decrementAndGet();
                if (!channelFuture.isSuccess()) {
                    logger.debug("{}, stream {}, Error writing request {}", new Object[]{Connection.this, request.getStreamId(), request});
                    Connection.this.dispatcher.removeHandler(responseHandler, true);
                    TransportException transportException = channelFuture.cause() instanceof ClosedChannelException ? new TransportException(Connection.this.address, "Error writing: Closed channel") : new TransportException(Connection.this.address, "Error writing", channelFuture.cause());
                    long l = System.nanoTime() - responseHandler.startTime;
                    ListeningExecutorService listeningExecutorService = ((Connection)Connection.this).factory.manager.executor;
                    if (!listeningExecutorService.isShutdown()) {
                        listeningExecutorService.execute(new Runnable((ConnectionException)transportException, l){
                            final /* synthetic */ ConnectionException val$ce;
                            final /* synthetic */ long val$latency;
                            {
                                this.val$ce = connectionException;
                                this.val$latency = l;
                            }

                            @Override
                            public void run() {
                                responseHandler.callback.onException(Connection.this, (Exception)((Object)Connection.this.defunct(this.val$ce)), this.val$latency, responseHandler.retryCount);
                            }
                        });
                    }
                } else {
                    logger.trace("{}, stream {}, request sent successfully", (Object)Connection.this, (Object)request.getStreamId());
                }
            }
        };
    }

    boolean hasOwner() {
        return this.ownerRef.get() != null;
    }

    boolean setOwner(Owner owner) {
        return this.ownerRef.compareAndSet(null, owner);
    }

    void release() {
        Owner owner = this.ownerRef.get();
        if (owner instanceof HostConnectionPool) {
            ((HostConnectionPool)owner).returnConnection(this);
        }
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public CloseFuture closeAsync() {
        boolean bl;
        Host host;
        ConnectionCloseFuture connectionCloseFuture = new ConnectionCloseFuture();
        if (!this.closeFuture.compareAndSet(null, connectionCloseFuture)) {
            return this.closeFuture.get();
        }
        logger.debug("{} closing connection", (Object)this);
        if (this.signaled.compareAndSet(false, true) && (host = this.factory.manager.metadata.getHost(this.address)) != null) {
            host.convictionPolicy.signalConnectionClosed(this);
        }
        if (!(bl = this.tryTerminate(false))) {
            long l = System.currentTimeMillis() + 2L * this.factory.getReadTimeoutMillis();
            this.factory.reaper.register(this, l);
        }
        return connectionCloseFuture;
    }

    boolean tryTerminate(boolean bl) {
        assert (this.isClosed());
        ConnectionCloseFuture connectionCloseFuture = this.closeFuture.get();
        if (connectionCloseFuture.isDone()) {
            logger.debug("{} has already terminated", (Object)this);
            return true;
        }
        if (bl || this.dispatcher.pending.isEmpty()) {
            if (bl) {
                logger.warn("Forcing termination of {}. This should not happen and is likely a bug, please report.", (Object)this);
            }
            connectionCloseFuture.force();
            return true;
        }
        logger.debug("Not terminating {}: there are still pending requests", (Object)this);
        return false;
    }

    public String toString() {
        return String.format("Connection[%s, inFlight=%d, closed=%b]", this.name, this.inFlight.get(), this.isClosed());
    }

    private void flush(FlushItem flushItem) {
        Flusher flusher;
        EventLoop eventLoop = flushItem.channel.eventLoop();
        Flusher flusher2 = (Flusher)flusherLookup.get(eventLoop);
        if (flusher2 == null && (flusher = flusherLookup.putIfAbsent(eventLoop, flusher2 = new Flusher(eventLoop))) != null) {
            flusher2 = flusher;
        }
        flusher2.queued.add(flushItem);
        flusher2.start();
    }

    static interface Owner {
        public void onConnectionDefunct(Connection var1);
    }

    private static class Initializer
    extends ChannelInitializer<SocketChannel> {
        private static final Message.ProtocolDecoder messageDecoder = new Message.ProtocolDecoder();
        private static final Message.ProtocolEncoder messageEncoderV1 = new Message.ProtocolEncoder(ProtocolVersion.V1);
        private static final Message.ProtocolEncoder messageEncoderV2 = new Message.ProtocolEncoder(ProtocolVersion.V2);
        private static final Message.ProtocolEncoder messageEncoderV3 = new Message.ProtocolEncoder(ProtocolVersion.V3);
        private static final Frame.Encoder frameEncoder = new Frame.Encoder();
        private final ProtocolVersion protocolVersion;
        private final Connection connection;
        private final FrameCompressor compressor;
        private final SSLOptions sslOptions;
        private final NettyOptions nettyOptions;
        private final ChannelHandler idleStateHandler;

        public Initializer(Connection connection, ProtocolVersion protocolVersion, FrameCompressor frameCompressor, SSLOptions sSLOptions, int n, NettyOptions nettyOptions) {
            this.connection = connection;
            this.protocolVersion = protocolVersion;
            this.compressor = frameCompressor;
            this.sslOptions = sSLOptions;
            this.nettyOptions = nettyOptions;
            this.idleStateHandler = new IdleStateHandler(0, 0, n);
        }

        protected void initChannel(SocketChannel socketChannel) throws Exception {
            ChannelPipeline channelPipeline = socketChannel.pipeline();
            if (this.sslOptions != null) {
                SSLEngine sSLEngine = this.sslOptions.context.createSSLEngine();
                sSLEngine.setUseClientMode(true);
                sSLEngine.setEnabledCipherSuites(this.sslOptions.cipherSuites);
                SslHandler sslHandler = new SslHandler(sSLEngine);
                channelPipeline.addLast("ssl", (ChannelHandler)sslHandler);
            }
            channelPipeline.addLast("frameDecoder", (ChannelHandler)new Frame.Decoder());
            channelPipeline.addLast("frameEncoder", (ChannelHandler)frameEncoder);
            if (this.compressor != null) {
                channelPipeline.addLast("frameDecompressor", (ChannelHandler)new Frame.Decompressor(this.compressor));
                channelPipeline.addLast("frameCompressor", (ChannelHandler)new Frame.Compressor(this.compressor));
            }
            channelPipeline.addLast("messageDecoder", (ChannelHandler)messageDecoder);
            channelPipeline.addLast("messageEncoder", (ChannelHandler)this.messageEncoderFor(this.protocolVersion));
            channelPipeline.addLast("idleStateHandler", this.idleStateHandler);
            channelPipeline.addLast("dispatcher", (ChannelHandler)this.connection.dispatcher);
            this.nettyOptions.afterChannelInitialized(socketChannel);
        }

        private Message.ProtocolEncoder messageEncoderFor(ProtocolVersion protocolVersion) {
            switch (protocolVersion) {
                case V1: {
                    return messageEncoderV1;
                }
                case V2: {
                    return messageEncoderV2;
                }
                case V3: {
                    return messageEncoderV3;
                }
            }
            throw new DriverInternalError("Unsupported protocol version " + this.protocolVersion);
        }
    }

    public static interface DefaultResponseHandler {
        public void handle(Message.Response var1);
    }

    static class ResponseHandler {
        public final Connection connection;
        public final int streamId;
        public final ResponseCallback callback;
        public final int retryCount;
        private final long startTime;
        private volatile Timeout timeout;
        private final AtomicBoolean isCancelled = new AtomicBoolean();

        public ResponseHandler(Connection connection, ResponseCallback responseCallback) throws BusyConnectionException {
            this.connection = connection;
            this.streamId = connection.dispatcher.streamIdHandler.next();
            this.callback = responseCallback;
            this.retryCount = responseCallback.retryCount();
            this.startTime = System.nanoTime();
        }

        void startTimeout() {
            long l = this.connection.factory.getReadTimeoutMillis();
            this.timeout = l <= 0L ? null : ((Connection)this.connection).factory.timer.newTimeout(this.onTimeoutTask(), l, TimeUnit.MILLISECONDS);
        }

        void cancelTimeout() {
            if (this.timeout != null) {
                this.timeout.cancel();
            }
        }

        public boolean cancelHandler() {
            if (!this.isCancelled.compareAndSet(false, true)) {
                return false;
            }
            this.connection.dispatcher.removeHandler(this, false);
            return true;
        }

        private TimerTask onTimeoutTask() {
            return new TimerTask(){

                public void run(Timeout timeout) {
                    if (callback.onTimeout(connection, System.nanoTime() - startTime, retryCount)) {
                        this.cancelHandler();
                    }
                }
            };
        }
    }

    static interface ResponseCallback {
        public Message.Request request();

        public int retryCount();

        public void onSet(Connection var1, Message.Response var2, long var3, int var5);

        public void onException(Connection var1, Exception var2, long var3, int var5);

        public boolean onTimeout(Connection var1, long var2, int var4);
    }

    static class Future
    extends AbstractFuture<Message.Response>
    implements RequestHandler.Callback {
        private final Message.Request request;
        private volatile InetSocketAddress address;

        public Future(Message.Request request) {
            this.request = request;
        }

        public void register(RequestHandler requestHandler) {
        }

        public Message.Request request() {
            return this.request;
        }

        public int retryCount() {
            return 0;
        }

        public void onSet(Connection connection, Message.Response response, ExecutionInfo executionInfo, Statement statement, long l) {
            this.onSet(connection, response, l, 0);
        }

        public void onSet(Connection connection, Message.Response response, long l, int n) {
            this.address = connection.address;
            super.set((Object)response);
        }

        public void onException(Connection connection, Exception exception, long l, int n) {
            if (connection != null) {
                this.address = connection.address;
            }
            super.setException((Throwable)exception);
        }

        public boolean onTimeout(Connection connection, long l, int n) {
            assert (connection != null);
            this.address = connection.address;
            return super.setException((Throwable)new OperationTimedOutException(connection.address));
        }

        public InetSocketAddress getAddress() {
            return this.address;
        }
    }

    private class ConnectionCloseFuture
    extends CloseFuture {
        private ConnectionCloseFuture() {
        }

        public ConnectionCloseFuture force() {
            if (Connection.this.channel == null) {
                this.set(null);
                return this;
            }
            Connection.this.dispatcher.errorOutAllHandler((ConnectionException)new TransportException(Connection.this.address, "Connection has been closed"));
            ChannelFuture channelFuture = Connection.this.channel.close();
            channelFuture.addListener((GenericFutureListener)new ChannelFutureListener(){

                public void operationComplete(ChannelFuture channelFuture) {
                    Connection.this.factory.allChannels.remove((Object)Connection.this.channel);
                    if (channelFuture.cause() != null) {
                        logger.warn("Error closing channel", channelFuture.cause());
                        ConnectionCloseFuture.this.setException(channelFuture.cause());
                    } else {
                        ConnectionCloseFuture.this.set(null);
                    }
                }
            });
            return this;
        }
    }

    private class ChannelCloseListener
    implements ChannelFutureListener {
        private ChannelCloseListener() {
        }

        public void operationComplete(ChannelFuture channelFuture) throws Exception {
            if (!Connection.this.isInitialized || Connection.this.isClosed()) {
                Connection.this.dispatcher.errorOutAllHandler((ConnectionException)new TransportException(Connection.this.address, "Channel has been closed"));
                Connection.this.closeAsync().force();
            } else {
                Connection.this.defunct(new TransportException(Connection.this.address, "Channel has been closed"));
            }
        }
    }

    class Dispatcher
    extends SimpleChannelInboundHandler<Message.Response> {
        public final StreamIdGenerator streamIdHandler;
        private final ConcurrentMap<Integer, ResponseHandler> pending = new ConcurrentHashMap<Integer, ResponseHandler>();

        Dispatcher() {
            ProtocolVersion protocolVersion = ((Connection)Connection.this).factory.protocolVersion;
            if (protocolVersion == null) {
                protocolVersion = ProtocolVersion.V2;
            }
            this.streamIdHandler = StreamIdGenerator.newInstance((ProtocolVersion)protocolVersion);
        }

        public void add(ResponseHandler responseHandler) {
            ResponseHandler responseHandler2 = this.pending.put(responseHandler.streamId, responseHandler);
            assert (responseHandler2 == null);
        }

        public void removeHandler(ResponseHandler responseHandler, boolean bl) {
            boolean bl2;
            if (!bl) {
                this.streamIdHandler.mark(responseHandler.streamId);
            }
            if (!(bl2 = this.pending.remove(responseHandler.streamId, responseHandler))) {
                if (!bl) {
                    this.streamIdHandler.unmark(responseHandler.streamId);
                }
                return;
            }
            responseHandler.cancelTimeout();
            if (bl) {
                this.streamIdHandler.release(responseHandler.streamId);
            }
            if (Connection.this.isClosed()) {
                Connection.this.tryTerminate(false);
            }
        }

        protected void channelRead0(ChannelHandlerContext channelHandlerContext, Message.Response response) throws Exception {
            int n = response.getStreamId();
            if (logger.isTraceEnabled()) {
                logger.trace("{}, stream {}, received: {}", new Object[]{Connection.this, n, this.asDebugString(response)});
            }
            if (n < 0) {
                ((Connection)Connection.this).factory.defaultHandler.handle(response);
                return;
            }
            ResponseHandler responseHandler = (ResponseHandler)this.pending.remove(n);
            this.streamIdHandler.release(n);
            if (responseHandler == null) {
                this.streamIdHandler.unmark(n);
                if (logger.isDebugEnabled()) {
                    logger.debug("{} Response received on stream {} but no handler set anymore (either the request has timed out or it was closed due to another error). Received message is {}", new Object[]{Connection.this, n, this.asDebugString(response)});
                }
                return;
            }
            responseHandler.cancelTimeout();
            responseHandler.callback.onSet(Connection.this, response, System.nanoTime() - responseHandler.startTime, responseHandler.retryCount);
            if (Connection.this.isClosed()) {
                Connection.this.tryTerminate(false);
            }
        }

        public void userEventTriggered(ChannelHandlerContext channelHandlerContext, Object object) throws Exception {
            if (!Connection.this.isClosed() && object instanceof IdleStateEvent && ((IdleStateEvent)object).state() == IdleState.ALL_IDLE) {
                logger.debug("{} was inactive for {} seconds, sending heartbeat", (Object)Connection.this, (Object)((Connection)Connection.this).factory.configuration.getPoolingOptions().getHeartbeatIntervalSeconds());
                Connection.this.write(HEARTBEAT_CALLBACK);
            }
        }

        private String asDebugString(Object object) {
            if (object == null) {
                return "null";
            }
            String string = object.toString();
            if (string.length() < 500) {
                return string;
            }
            return string.substring(0, 500) + "... [message of size " + string.length() + " truncated]";
        }

        public void exceptionCaught(ChannelHandlerContext channelHandlerContext, Throwable throwable) throws Exception {
            if (logger.isDebugEnabled()) {
                logger.debug(String.format("%s connection error", Connection.this), throwable);
            }
            if (Connection.this.writer.get() > 0) {
                return;
            }
            Connection.this.defunct(new TransportException(Connection.this.address, String.format("Unexpected exception triggered (%s)", throwable), throwable));
        }

        public void errorOutAllHandler(ConnectionException connectionException) {
            Iterator iterator = this.pending.values().iterator();
            while (iterator.hasNext()) {
                ResponseHandler responseHandler = (ResponseHandler)iterator.next();
                responseHandler.cancelTimeout();
                responseHandler.callback.onException(Connection.this, (Exception)((Object)connectionException), System.nanoTime() - responseHandler.startTime, responseHandler.retryCount);
                iterator.remove();
            }
        }
    }

    private static class FlushItem {
        final Channel channel;
        final Object request;
        final ChannelFutureListener listener;

        private FlushItem(Channel channel, Object object, ChannelFutureListener channelFutureListener) {
            this.channel = channel;
            this.request = object;
            this.listener = channelFutureListener;
        }
    }

    private static final class Flusher
    implements Runnable {
        final WeakReference<EventLoop> eventLoopRef;
        final Queue<FlushItem> queued = new ConcurrentLinkedQueue<FlushItem>();
        final AtomicBoolean running = new AtomicBoolean(false);
        final HashSet<Channel> channels = new HashSet();
        int runsWithNoWork = 0;

        private Flusher(EventLoop eventLoop) {
            this.eventLoopRef = new WeakReference<EventLoop>(eventLoop);
        }

        void start() {
            EventLoop eventLoop;
            if (!this.running.get() && this.running.compareAndSet(false, true) && (eventLoop = (EventLoop)this.eventLoopRef.get()) != null) {
                eventLoop.execute((Runnable)this);
            }
        }

        @Override
        public void run() {
            Object object;
            FlushItem flushItem;
            boolean bl = false;
            while (null != (flushItem = this.queued.poll())) {
                object = flushItem.channel;
                if (!object.isActive()) continue;
                this.channels.add((Channel)object);
                object.write(flushItem.request).addListener((GenericFutureListener)flushItem.listener);
                bl = true;
            }
            for (Channel channel : this.channels) {
                channel.flush();
            }
            this.channels.clear();
            if (bl) {
                this.runsWithNoWork = 0;
            } else if (++this.runsWithNoWork > 5) {
                this.running.set(false);
                if (this.queued.isEmpty() || !this.running.compareAndSet(false, true)) {
                    return;
                }
            }
            object = (EventLoop)this.eventLoopRef.get();
            if (object != null && !object.isShuttingDown()) {
                object.schedule((Runnable)this, 10000L, TimeUnit.NANOSECONDS);
            }
        }
    }

    public static class Factory {
        public final Timer timer;
        final EventLoopGroup eventLoopGroup;
        private final Class<? extends Channel> channelClass;
        private final ChannelGroup allChannels = new DefaultChannelGroup((EventExecutor)GlobalEventExecutor.INSTANCE);
        private final ConcurrentMap<Host, AtomicInteger> idGenerators = new ConcurrentHashMap<Host, AtomicInteger>();
        public final DefaultResponseHandler defaultHandler;
        final Cluster.Manager manager;
        final Cluster.ConnectionReaper reaper;
        public final Configuration configuration;
        public final AuthProvider authProvider;
        private volatile boolean isShutdown;
        volatile ProtocolVersion protocolVersion;
        private final NettyOptions nettyOptions;

        Factory(Cluster.Manager manager, Configuration configuration) {
            this.defaultHandler = manager;
            this.manager = manager;
            this.reaper = manager.reaper;
            this.configuration = configuration;
            this.authProvider = configuration.getProtocolOptions().getAuthProvider();
            this.protocolVersion = configuration.getProtocolOptions().initialProtocolVersion;
            this.nettyOptions = configuration.getNettyOptions();
            this.eventLoopGroup = this.nettyOptions.eventLoopGroup(manager.threadFactory("nio-worker"));
            this.channelClass = this.nettyOptions.channelClass();
            this.timer = this.nettyOptions.timer(manager.threadFactory("timeouter"));
        }

        public int getPort() {
            return this.configuration.getProtocolOptions().getPort();
        }

        public Connection open(Host host) throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            InetSocketAddress inetSocketAddress = host.getSocketAddress();
            if (this.isShutdown) {
                throw new ConnectionException(inetSocketAddress, "Connection factory is shut down");
            }
            host.convictionPolicy.signalConnectionsOpening(1);
            Connection connection = new Connection(this.buildConnectionName(host), inetSocketAddress, this);
            try {
                connection.initAsync().get();
                return connection;
            }
            catch (ExecutionException executionException) {
                throw Factory.launderAsyncInitException(executionException);
            }
        }

        public Connection open(HostConnectionPool hostConnectionPool) throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            hostConnectionPool.host.convictionPolicy.signalConnectionsOpening(1);
            Connection connection = new Connection(this.buildConnectionName(hostConnectionPool.host), hostConnectionPool.host.getSocketAddress(), this, hostConnectionPool);
            try {
                connection.initAsync().get();
                return connection;
            }
            catch (ExecutionException executionException) {
                throw Factory.launderAsyncInitException(executionException);
            }
        }

        public List<Connection> newConnections(HostConnectionPool hostConnectionPool, int n) {
            hostConnectionPool.host.convictionPolicy.signalConnectionsOpening(n);
            ArrayList arrayList = Lists.newArrayListWithCapacity((int)n);
            for (int i = 0; i < n; ++i) {
                arrayList.add(new Connection(this.buildConnectionName(hostConnectionPool.host), hostConnectionPool.host.getSocketAddress(), this, hostConnectionPool));
            }
            return arrayList;
        }

        private String buildConnectionName(Host host) {
            return host.getSocketAddress().toString() + '-' + this.getIdGenerator(host).getAndIncrement();
        }

        static RuntimeException launderAsyncInitException(ExecutionException executionException) throws ConnectionException, InterruptedException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
            Throwable throwable = executionException.getCause();
            if (throwable instanceof ConnectionException) {
                throw (ConnectionException)throwable;
            }
            if (throwable instanceof InterruptedException) {
                throw (InterruptedException)throwable;
            }
            if (throwable instanceof UnsupportedProtocolVersionException) {
                throw (UnsupportedProtocolVersionException)throwable;
            }
            if (throwable instanceof ClusterNameMismatchException) {
                throw (ClusterNameMismatchException)throwable;
            }
            if (throwable instanceof DriverException) {
                throw (DriverException)throwable;
            }
            return new RuntimeException("Unexpected exception during connection initialization", throwable);
        }

        private AtomicInteger getIdGenerator(Host host) {
            AtomicInteger atomicInteger;
            AtomicInteger atomicInteger2 = (AtomicInteger)this.idGenerators.get(host);
            if (atomicInteger2 == null && (atomicInteger = this.idGenerators.putIfAbsent(host, atomicInteger2 = new AtomicInteger(1))) != null) {
                atomicInteger2 = atomicInteger;
            }
            return atomicInteger2;
        }

        public long getReadTimeoutMillis() {
            return this.configuration.getSocketOptions().getReadTimeoutMillis();
        }

        private Bootstrap newBootstrap() {
            Integer n;
            Integer n2;
            Boolean bl;
            Integer n3;
            Boolean bl2;
            Bootstrap bootstrap = new Bootstrap();
            ((Bootstrap)bootstrap.group(this.eventLoopGroup)).channel(this.channelClass);
            SocketOptions socketOptions = this.configuration.getSocketOptions();
            bootstrap.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, (Object)socketOptions.getConnectTimeoutMillis());
            Boolean bl3 = socketOptions.getKeepAlive();
            if (bl3 != null) {
                bootstrap.option(ChannelOption.SO_KEEPALIVE, (Object)bl3);
            }
            if ((bl2 = socketOptions.getReuseAddress()) != null) {
                bootstrap.option(ChannelOption.SO_REUSEADDR, (Object)bl2);
            }
            if ((n3 = socketOptions.getSoLinger()) != null) {
                bootstrap.option(ChannelOption.SO_LINGER, (Object)n3);
            }
            if ((bl = socketOptions.getTcpNoDelay()) != null) {
                bootstrap.option(ChannelOption.TCP_NODELAY, (Object)bl);
            }
            if ((n2 = socketOptions.getReceiveBufferSize()) != null) {
                bootstrap.option(ChannelOption.SO_RCVBUF, (Object)n2);
            }
            if ((n = socketOptions.getSendBufferSize()) != null) {
                bootstrap.option(ChannelOption.SO_SNDBUF, (Object)n);
            }
            this.nettyOptions.afterBootstrapInitialized(bootstrap);
            return bootstrap;
        }

        public void shutdown() {
            this.isShutdown = true;
            this.allChannels.close().awaitUninterruptibly();
            this.nettyOptions.onClusterClose(this.eventLoopGroup);
            this.nettyOptions.onClusterClose(this.timer);
        }
    }

    static enum State {
        OPEN,
        TRASHED,
        RESURRECTING,
        GONE;

    }
}

