//[okhttp](../../../index.md)/[okhttp3](../index.md)/[ResponseBody](index.md)

# ResponseBody

[common]\
expect abstract class [ResponseBody](index.md)

A one-shot stream from the origin server to the client application with the raw bytes of the response body. Each response body is supported by an active connection to the webserver. This imposes both obligations and limits on the client application.

###  The response body must be closed.

Each response body is backed by a limited resource like a socket (live network responses) or an open file (for cached responses). Failing to close the response body will leak resources and may ultimately cause the application to slow down or crash.

Both this class and [Response](../-response/index.md) implement Closeable. Closing a response simply closes its response body. If you invoke Call.execute or implement [Callback.onResponse](../-callback/on-response.md) you must close this body by calling any of the following methods:

- 
   Response.close()
- 
   Response.body().close()
- 
   Response.body().source().close()
- 
   Response.body().charStream().close()
- 
   Response.body().byteStream().close()
- 
   Response.body().bytes()
- 
   Response.body().string()

There is no benefit to invoking multiple close() methods for the same response body.

For synchronous calls, the easiest way to make sure a response body is closed is with a try block. With this structure the compiler inserts an implicit finally clause that calls [close()](../-response/close.md) for you.

Call call = client.newCall(request);\
try (Response response = call.execute()) {\
... // Use the response.\
}

You can use a similar block for asynchronous calls:

Call call = client.newCall(request);\
call.enqueue(new Callback() {\
  public void onResponse(Call call, Response response) throws IOException {\
    try (ResponseBody responseBody = response.body()) {\
    ... // Use the response.\
    }\
  }\
\
  public void onFailure(Call call, IOException e) {\
  ... // Handle the failure.\
  }\
});

These examples will not work if you're consuming the response body on another thread. In such cases the consuming thread must call [close](close.md) when it has finished reading the response body.

###  The response body can be consumed only once.

This class may be used to stream very large responses. For example, it is possible to use this class to read a response that is larger than the entire memory allocated to the current process. It can even stream a response larger than the total storage on the current device, which is a common requirement for video streaming applications.

Because this class does not buffer the full response in memory, the application may not re-read the bytes of the response. Use this one shot to read the entire response into memory with [bytes](bytes.md) or [string](string.md). Or stream the response with either [source](source.md), byteStream, or charStream.

[jvm]\
actual abstract class [ResponseBody](index.md) : [Closeable](https://docs.oracle.com/javase/8/docs/api/java/io/Closeable.html)

[nonJvm]\
actual abstract class [ResponseBody](index.md) : [Closeable](https://square.github.io/okio/2.x/okio/okio/-closeable/index.html)

## Constructors

| | |
|---|---|
| [ResponseBody](-response-body.md) | [jvm, nonJvm]<br>fun [ResponseBody](-response-body.md)()<br>[common]<br>expect fun [ResponseBody](-response-body.md)() |

## Types

| Name | Summary |
|---|---|
| [Companion](-companion/index.md) | [common, jvm, nonJvm]<br>[common]<br>expect object [Companion](-companion/index.md)<br>[jvm, nonJvm]<br>actual object [Companion](-companion/index.md) |

## Functions

| Name | Summary |
|---|---|
| [bytes](bytes.md) | [common]<br>expect fun [bytes](bytes.md)(): [ByteArray](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-byte-array/index.html)<br>Returns the response as a byte array.<br>[jvm, nonJvm]<br>[jvm, nonJvm]<br>actual fun [bytes](bytes.md)(): [ByteArray](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-byte-array/index.html) |
| [byteStream](byte-stream.md) | [jvm]<br>fun [byteStream](byte-stream.md)(): [InputStream](https://docs.oracle.com/javase/8/docs/api/java/io/InputStream.html) |
| [byteString](byte-string.md) | [common]<br>expect fun [byteString](byte-string.md)(): &lt;ERROR CLASS&gt;<br>Returns the response as a ByteString.<br>[jvm, nonJvm]<br>[jvm, nonJvm]<br>actual fun [byteString](byte-string.md)(): &lt;ERROR CLASS&gt; |
| [charStream](char-stream.md) | [jvm]<br>fun [charStream](char-stream.md)(): [Reader](https://docs.oracle.com/javase/8/docs/api/java/io/Reader.html)<br>Returns the response as a character stream. |
| [close](close.md) | [common, jvm, nonJvm]<br>[common]<br>expect open fun [close](close.md)()<br>[jvm, nonJvm]<br>actual open override fun [close](close.md)() |
| [contentLength](content-length.md) | [common]<br>expect abstract fun [contentLength](content-length.md)(): [Long](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-long/index.html)<br>Returns the number of bytes in that will returned by [bytes](bytes.md), or byteStream, or -1 if unknown.<br>[jvm, nonJvm]<br>[jvm, nonJvm]<br>actual abstract fun [contentLength](content-length.md)(): [Long](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-long/index.html) |
| [contentType](content-type.md) | [common, jvm, nonJvm]<br>[common]<br>expect abstract fun [contentType](content-type.md)(): [MediaType](../-media-type/index.md)?<br>[jvm, nonJvm]<br>actual abstract fun [contentType](content-type.md)(): [MediaType](../-media-type/index.md)? |
| [source](source.md) | [common, jvm, nonJvm]<br>[common]<br>expect abstract fun [source](source.md)(): &lt;ERROR CLASS&gt;<br>[jvm, nonJvm]<br>actual abstract fun [source](source.md)(): &lt;ERROR CLASS&gt; |
| [string](string.md) | [common]<br>expect fun [string](string.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)<br>Returns the response as a string.<br>[jvm, nonJvm]<br>[jvm, nonJvm]<br>actual fun [string](string.md)(): [String](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html) |
