
package com.squareup.square.api;

import com.squareup.square.ApiHelper;
import com.squareup.square.AuthManager;
import com.squareup.square.Configuration;
import com.squareup.square.exceptions.ApiException;
import com.squareup.square.http.Headers;
import com.squareup.square.http.client.HttpCallback;
import com.squareup.square.http.client.HttpClient;
import com.squareup.square.http.client.HttpContext;
import com.squareup.square.http.request.HttpRequest;
import com.squareup.square.http.response.HttpResponse;
import com.squareup.square.http.response.HttpStringResponse;
import com.squareup.square.models.ListMerchantsResponse;
import com.squareup.square.models.RetrieveMerchantResponse;
import java.io.IOException;
import java.util.AbstractMap.SimpleEntry;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * This class lists all the endpoints of the groups.
 */
public final class DefaultMerchantsApi extends BaseApi implements MerchantsApi {

    /**
     * Initializes the controller.
     * @param config    Configurations added in client.
     * @param httpClient    Send HTTP requests and read the responses.
     * @param authManagers    Apply authorization to requests.
     */
    public DefaultMerchantsApi(Configuration config, HttpClient httpClient,
            Map<String, AuthManager> authManagers) {
        super(config, httpClient, authManagers);
    }

    /**
     * Initializes the controller with HTTPCallback.
     * @param config    Configurations added in client.
     * @param httpClient    Send HTTP requests and read the responses.
     * @param authManagers    Apply authorization to requests.
     * @param httpCallback    Callback to be called before and after the HTTP call.
     */
    public DefaultMerchantsApi(Configuration config, HttpClient httpClient,
            Map<String, AuthManager> authManagers, HttpCallback httpCallback) {
        super(config, httpClient, authManagers, httpCallback);
    }

    /**
     * Provides details about the merchant associated with a given access token. The access token
     * used to connect your application to a Square seller is associated with a single merchant.
     * That means that `ListMerchants` returns a list with a single `Merchant` object. You can
     * specify your personal access token to get your own merchant information or specify an OAuth
     * token to get the information for the merchant that granted your application access. If you
     * know the merchant ID, you can also use the [RetrieveMerchant]($e/Merchants/RetrieveMerchant)
     * endpoint to retrieve the merchant information.
     * @param  cursor  Optional parameter: The cursor generated by the previous response.
     * @return    Returns the ListMerchantsResponse response from the API call
     * @throws    ApiException    Represents error response from the server.
     * @throws    IOException    Signals that an I/O exception of some sort has occurred.
     */
    public ListMerchantsResponse listMerchants(
            final Integer cursor) throws ApiException, IOException {
        HttpRequest request = buildListMerchantsRequest(cursor);
        authManagers.get("global").apply(request);

        HttpResponse response = getClientInstance().execute(request, false);
        HttpContext context = new HttpContext(request, response);

        return handleListMerchantsResponse(context);
    }

    /**
     * Provides details about the merchant associated with a given access token. The access token
     * used to connect your application to a Square seller is associated with a single merchant.
     * That means that `ListMerchants` returns a list with a single `Merchant` object. You can
     * specify your personal access token to get your own merchant information or specify an OAuth
     * token to get the information for the merchant that granted your application access. If you
     * know the merchant ID, you can also use the [RetrieveMerchant]($e/Merchants/RetrieveMerchant)
     * endpoint to retrieve the merchant information.
     * @param  cursor  Optional parameter: The cursor generated by the previous response.
     * @return    Returns the ListMerchantsResponse response from the API call
     */
    public CompletableFuture<ListMerchantsResponse> listMerchantsAsync(
            final Integer cursor) {
        return makeHttpCallAsync(() -> buildListMerchantsRequest(cursor),
            req -> authManagers.get("global").applyAsync(req)
                .thenCompose(request -> getClientInstance()
                        .executeAsync(request, false)),
            context -> handleListMerchantsResponse(context));
    }

    /**
     * Builds the HttpRequest object for listMerchants.
     */
    private HttpRequest buildListMerchantsRequest(
            final Integer cursor) {
        //the base uri for api requests
        String baseUri = config.getBaseUri();

        //prepare query string for API call
        final StringBuilder queryBuilder = new StringBuilder(baseUri
                + "/v2/merchants");

        //load all query parameters
        Map<String, Object> queryParameters = new HashMap<>();
        queryParameters.put("cursor", cursor);

        //load all headers for the outgoing API request
        Headers headers = new Headers();
        headers.add("Square-Version", config.getSquareVersion());
        headers.add("user-agent", internalUserAgent);
        headers.add("accept", "application/json");
        headers.addAll(config.getAdditionalHeaders());

        //prepare and invoke the API call request to fetch the response
        HttpRequest request = getClientInstance().get(queryBuilder, headers, queryParameters,
                null);

        // Invoke the callback before request if its not null
        if (getHttpCallback() != null) {
            getHttpCallback().onBeforeRequest(request);
        }

        return request;
    }

    /**
     * Processes the response for listMerchants.
     * @return An object of type ListMerchantsResponse
     */
    private ListMerchantsResponse handleListMerchantsResponse(
            HttpContext context) throws ApiException, IOException {
        HttpResponse response = context.getResponse();

        //invoke the callback after response if its not null
        if (getHttpCallback() != null) {
            getHttpCallback().onAfterResponse(context);
        }

        //handle errors defined at the API level
        validateResponse(response, context);

        //extract result from the http response
        String responseBody = ((HttpStringResponse) response).getBody();
        ListMerchantsResponse result = ApiHelper.deserialize(responseBody,
                ListMerchantsResponse.class);

        result = result.toBuilder().httpContext(context).build();
        return result;
    }

    /**
     * Retrieves the `Merchant` object for the given `merchant_id`.
     * @param  merchantId  Required parameter: The ID of the merchant to retrieve. If the string
     *         "me" is supplied as the ID, then retrieve the merchant that is currently accessible
     *         to this call.
     * @return    Returns the RetrieveMerchantResponse response from the API call
     * @throws    ApiException    Represents error response from the server.
     * @throws    IOException    Signals that an I/O exception of some sort has occurred.
     */
    public RetrieveMerchantResponse retrieveMerchant(
            final String merchantId) throws ApiException, IOException {
        HttpRequest request = buildRetrieveMerchantRequest(merchantId);
        authManagers.get("global").apply(request);

        HttpResponse response = getClientInstance().execute(request, false);
        HttpContext context = new HttpContext(request, response);

        return handleRetrieveMerchantResponse(context);
    }

    /**
     * Retrieves the `Merchant` object for the given `merchant_id`.
     * @param  merchantId  Required parameter: The ID of the merchant to retrieve. If the string
     *         "me" is supplied as the ID, then retrieve the merchant that is currently accessible
     *         to this call.
     * @return    Returns the RetrieveMerchantResponse response from the API call
     */
    public CompletableFuture<RetrieveMerchantResponse> retrieveMerchantAsync(
            final String merchantId) {
        return makeHttpCallAsync(() -> buildRetrieveMerchantRequest(merchantId),
            req -> authManagers.get("global").applyAsync(req)
                .thenCompose(request -> getClientInstance()
                        .executeAsync(request, false)),
            context -> handleRetrieveMerchantResponse(context));
    }

    /**
     * Builds the HttpRequest object for retrieveMerchant.
     */
    private HttpRequest buildRetrieveMerchantRequest(
            final String merchantId) {
        //the base uri for api requests
        String baseUri = config.getBaseUri();

        //prepare query string for API call
        final StringBuilder queryBuilder = new StringBuilder(baseUri
                + "/v2/merchants/{merchant_id}");

        //process template parameters
        Map<String, SimpleEntry<Object, Boolean>> templateParameters = new HashMap<>();
        templateParameters.put("merchant_id",
                new SimpleEntry<Object, Boolean>(merchantId, true));
        ApiHelper.appendUrlWithTemplateParameters(queryBuilder, templateParameters);

        //load all headers for the outgoing API request
        Headers headers = new Headers();
        headers.add("Square-Version", config.getSquareVersion());
        headers.add("user-agent", internalUserAgent);
        headers.add("accept", "application/json");
        headers.addAll(config.getAdditionalHeaders());

        //prepare and invoke the API call request to fetch the response
        HttpRequest request = getClientInstance().get(queryBuilder, headers, null, null);

        // Invoke the callback before request if its not null
        if (getHttpCallback() != null) {
            getHttpCallback().onBeforeRequest(request);
        }

        return request;
    }

    /**
     * Processes the response for retrieveMerchant.
     * @return An object of type RetrieveMerchantResponse
     */
    private RetrieveMerchantResponse handleRetrieveMerchantResponse(
            HttpContext context) throws ApiException, IOException {
        HttpResponse response = context.getResponse();

        //invoke the callback after response if its not null
        if (getHttpCallback() != null) {
            getHttpCallback().onAfterResponse(context);
        }

        //handle errors defined at the API level
        validateResponse(response, context);

        //extract result from the http response
        String responseBody = ((HttpStringResponse) response).getBody();
        RetrieveMerchantResponse result = ApiHelper.deserialize(responseBody,
                RetrieveMerchantResponse.class);

        result = result.toBuilder().httpContext(context).build();
        return result;
    }

}