
package com.squareup.square.models;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;

/**
 * This is a model class for CatalogImage type.
 */
public class CatalogImage {
    private final String name;
    private final String url;
    private final String caption;
    private final String photoStudioOrderId;

    /**
     * Initialization constructor.
     * @param  name  String value for name.
     * @param  url  String value for url.
     * @param  caption  String value for caption.
     * @param  photoStudioOrderId  String value for photoStudioOrderId.
     */
    @JsonCreator
    public CatalogImage(
            @JsonProperty("name") String name,
            @JsonProperty("url") String url,
            @JsonProperty("caption") String caption,
            @JsonProperty("photo_studio_order_id") String photoStudioOrderId) {
        this.name = name;
        this.url = url;
        this.caption = caption;
        this.photoStudioOrderId = photoStudioOrderId;
    }

    /**
     * Getter for Name.
     * The internal name to identify this image in calls to the Square API. This is a searchable
     * attribute for use in applicable query filters using the
     * [SearchCatalogObjects]($e/Catalog/SearchCatalogObjects). It is not unique and should not be
     * shown in a buyer facing context.
     * @return Returns the String
     */
    @JsonGetter("name")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getName() {
        return name;
    }

    /**
     * Getter for Url.
     * The URL of this image, generated by Square after an image is uploaded using the
     * [CreateCatalogImage]($e/Catalog/CreateCatalogImage) endpoint. To modify the image, use the
     * UpdateCatalogImage endpoint. Do not change the URL field.
     * @return Returns the String
     */
    @JsonGetter("url")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getUrl() {
        return url;
    }

    /**
     * Getter for Caption.
     * A caption that describes what is shown in the image. Displayed in the Square Online Store.
     * This is a searchable attribute for use in applicable query filters using the
     * [SearchCatalogObjects]($e/Catalog/SearchCatalogObjects).
     * @return Returns the String
     */
    @JsonGetter("caption")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getCaption() {
        return caption;
    }

    /**
     * Getter for PhotoStudioOrderId.
     * The immutable order ID for this image object created by the Photo Studio service in Square
     * Online Store.
     * @return Returns the String
     */
    @JsonGetter("photo_studio_order_id")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getPhotoStudioOrderId() {
        return photoStudioOrderId;
    }

    @Override
    public int hashCode() {
        return Objects.hash(name, url, caption, photoStudioOrderId);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof CatalogImage)) {
            return false;
        }
        CatalogImage other = (CatalogImage) obj;
        return Objects.equals(name, other.name)
            && Objects.equals(url, other.url)
            && Objects.equals(caption, other.caption)
            && Objects.equals(photoStudioOrderId, other.photoStudioOrderId);
    }

    /**
     * Converts this CatalogImage into string format.
     * @return String representation of this class
     */
    @Override
    public String toString() {
        return "CatalogImage [" + "name=" + name + ", url=" + url + ", caption=" + caption
                + ", photoStudioOrderId=" + photoStudioOrderId + "]";
    }

    /**
     * Builds a new {@link CatalogImage.Builder} object.
     * Creates the instance with the state of the current model.
     * @return a new {@link CatalogImage.Builder} object
     */
    public Builder toBuilder() {
        Builder builder = new Builder()
                .name(getName())
                .url(getUrl())
                .caption(getCaption())
                .photoStudioOrderId(getPhotoStudioOrderId());
        return builder;
    }

    /**
     * Class to build instances of {@link CatalogImage}.
     */
    public static class Builder {
        private String name;
        private String url;
        private String caption;
        private String photoStudioOrderId;



        /**
         * Setter for name.
         * @param  name  String value for name.
         * @return Builder
         */
        public Builder name(String name) {
            this.name = name;
            return this;
        }

        /**
         * Setter for url.
         * @param  url  String value for url.
         * @return Builder
         */
        public Builder url(String url) {
            this.url = url;
            return this;
        }

        /**
         * Setter for caption.
         * @param  caption  String value for caption.
         * @return Builder
         */
        public Builder caption(String caption) {
            this.caption = caption;
            return this;
        }

        /**
         * Setter for photoStudioOrderId.
         * @param  photoStudioOrderId  String value for photoStudioOrderId.
         * @return Builder
         */
        public Builder photoStudioOrderId(String photoStudioOrderId) {
            this.photoStudioOrderId = photoStudioOrderId;
            return this;
        }

        /**
         * Builds a new {@link CatalogImage} object using the set fields.
         * @return {@link CatalogImage}
         */
        public CatalogImage build() {
            return new CatalogImage(name, url, caption, photoStudioOrderId);
        }
    }
}
