
package com.squareup.square.models;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.List;
import java.util.Objects;

/**
 * This is a model class for PaymentRefund type.
 */
public class PaymentRefund {
    private final String id;
    private final String status;
    private final String locationId;
    private final Money amountMoney;
    private final Money appFeeMoney;
    private final List<ProcessingFee> processingFee;
    private final String paymentId;
    private final String orderId;
    private final String reason;
    private final String createdAt;
    private final String updatedAt;
    private final String teamMemberId;

    /**
     * Initialization constructor.
     * @param  id  String value for id.
     * @param  amountMoney  Money value for amountMoney.
     * @param  status  String value for status.
     * @param  locationId  String value for locationId.
     * @param  appFeeMoney  Money value for appFeeMoney.
     * @param  processingFee  List of ProcessingFee value for processingFee.
     * @param  paymentId  String value for paymentId.
     * @param  orderId  String value for orderId.
     * @param  reason  String value for reason.
     * @param  createdAt  String value for createdAt.
     * @param  updatedAt  String value for updatedAt.
     * @param  teamMemberId  String value for teamMemberId.
     */
    @JsonCreator
    public PaymentRefund(
            @JsonProperty("id") String id,
            @JsonProperty("amount_money") Money amountMoney,
            @JsonProperty("status") String status,
            @JsonProperty("location_id") String locationId,
            @JsonProperty("app_fee_money") Money appFeeMoney,
            @JsonProperty("processing_fee") List<ProcessingFee> processingFee,
            @JsonProperty("payment_id") String paymentId,
            @JsonProperty("order_id") String orderId,
            @JsonProperty("reason") String reason,
            @JsonProperty("created_at") String createdAt,
            @JsonProperty("updated_at") String updatedAt,
            @JsonProperty("team_member_id") String teamMemberId) {
        this.id = id;
        this.status = status;
        this.locationId = locationId;
        this.amountMoney = amountMoney;
        this.appFeeMoney = appFeeMoney;
        this.processingFee = processingFee;
        this.paymentId = paymentId;
        this.orderId = orderId;
        this.reason = reason;
        this.createdAt = createdAt;
        this.updatedAt = updatedAt;
        this.teamMemberId = teamMemberId;
    }

    /**
     * Getter for Id.
     * The unique ID for this refund, generated by Square.
     * @return Returns the String
     */
    @JsonGetter("id")
    public String getId() {
        return id;
    }

    /**
     * Getter for Status.
     * The refund's status: - `PENDING` - Awaiting approval. - `COMPLETED` - Successfully completed.
     * - `REJECTED` - The refund was rejected. - `FAILED` - An error occurred.
     * @return Returns the String
     */
    @JsonGetter("status")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getStatus() {
        return status;
    }

    /**
     * Getter for LocationId.
     * The location ID associated with the payment this refund is attached to.
     * @return Returns the String
     */
    @JsonGetter("location_id")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getLocationId() {
        return locationId;
    }

    /**
     * Getter for AmountMoney.
     * Represents an amount of money. `Money` fields can be signed or unsigned. Fields that do not
     * explicitly define whether they are signed or unsigned are considered unsigned and can only
     * hold positive amounts. For signed fields, the sign of the value indicates the purpose of the
     * money transfer. See [Working with Monetary
     * Amounts](https://developer.squareup.com/docs/build-basics/working-with-monetary-amounts) for
     * more information.
     * @return Returns the Money
     */
    @JsonGetter("amount_money")
    public Money getAmountMoney() {
        return amountMoney;
    }

    /**
     * Getter for AppFeeMoney.
     * Represents an amount of money. `Money` fields can be signed or unsigned. Fields that do not
     * explicitly define whether they are signed or unsigned are considered unsigned and can only
     * hold positive amounts. For signed fields, the sign of the value indicates the purpose of the
     * money transfer. See [Working with Monetary
     * Amounts](https://developer.squareup.com/docs/build-basics/working-with-monetary-amounts) for
     * more information.
     * @return Returns the Money
     */
    @JsonGetter("app_fee_money")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public Money getAppFeeMoney() {
        return appFeeMoney;
    }

    /**
     * Getter for ProcessingFee.
     * Processing fees and fee adjustments assessed by Square for this refund.
     * @return Returns the List of ProcessingFee
     */
    @JsonGetter("processing_fee")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public List<ProcessingFee> getProcessingFee() {
        return processingFee;
    }

    /**
     * Getter for PaymentId.
     * The ID of the payment associated with this refund.
     * @return Returns the String
     */
    @JsonGetter("payment_id")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getPaymentId() {
        return paymentId;
    }

    /**
     * Getter for OrderId.
     * The ID of the order associated with the refund.
     * @return Returns the String
     */
    @JsonGetter("order_id")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getOrderId() {
        return orderId;
    }

    /**
     * Getter for Reason.
     * The reason for the refund.
     * @return Returns the String
     */
    @JsonGetter("reason")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getReason() {
        return reason;
    }

    /**
     * Getter for CreatedAt.
     * The timestamp of when the refund was created, in RFC 3339 format.
     * @return Returns the String
     */
    @JsonGetter("created_at")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getCreatedAt() {
        return createdAt;
    }

    /**
     * Getter for UpdatedAt.
     * The timestamp of when the refund was last updated, in RFC 3339 format.
     * @return Returns the String
     */
    @JsonGetter("updated_at")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getUpdatedAt() {
        return updatedAt;
    }

    /**
     * Getter for TeamMemberId.
     * An optional ID of the team member associated with taking the payment.
     * @return Returns the String
     */
    @JsonGetter("team_member_id")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public String getTeamMemberId() {
        return teamMemberId;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, status, locationId, amountMoney, appFeeMoney, processingFee,
                paymentId, orderId, reason, createdAt, updatedAt, teamMemberId);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof PaymentRefund)) {
            return false;
        }
        PaymentRefund other = (PaymentRefund) obj;
        return Objects.equals(id, other.id)
            && Objects.equals(status, other.status)
            && Objects.equals(locationId, other.locationId)
            && Objects.equals(amountMoney, other.amountMoney)
            && Objects.equals(appFeeMoney, other.appFeeMoney)
            && Objects.equals(processingFee, other.processingFee)
            && Objects.equals(paymentId, other.paymentId)
            && Objects.equals(orderId, other.orderId)
            && Objects.equals(reason, other.reason)
            && Objects.equals(createdAt, other.createdAt)
            && Objects.equals(updatedAt, other.updatedAt)
            && Objects.equals(teamMemberId, other.teamMemberId);
    }

    /**
     * Converts this PaymentRefund into string format.
     * @return String representation of this class
     */
    @Override
    public String toString() {
        return "PaymentRefund [" + "id=" + id + ", amountMoney=" + amountMoney + ", status="
                + status + ", locationId=" + locationId + ", appFeeMoney=" + appFeeMoney
                + ", processingFee=" + processingFee + ", paymentId=" + paymentId + ", orderId="
                + orderId + ", reason=" + reason + ", createdAt=" + createdAt + ", updatedAt="
                + updatedAt + ", teamMemberId=" + teamMemberId + "]";
    }

    /**
     * Builds a new {@link PaymentRefund.Builder} object.
     * Creates the instance with the state of the current model.
     * @return a new {@link PaymentRefund.Builder} object
     */
    public Builder toBuilder() {
        Builder builder = new Builder(id, amountMoney)
                .status(getStatus())
                .locationId(getLocationId())
                .appFeeMoney(getAppFeeMoney())
                .processingFee(getProcessingFee())
                .paymentId(getPaymentId())
                .orderId(getOrderId())
                .reason(getReason())
                .createdAt(getCreatedAt())
                .updatedAt(getUpdatedAt())
                .teamMemberId(getTeamMemberId());
        return builder;
    }

    /**
     * Class to build instances of {@link PaymentRefund}.
     */
    public static class Builder {
        private String id;
        private Money amountMoney;
        private String status;
        private String locationId;
        private Money appFeeMoney;
        private List<ProcessingFee> processingFee;
        private String paymentId;
        private String orderId;
        private String reason;
        private String createdAt;
        private String updatedAt;
        private String teamMemberId;

        /**
         * Initialization constructor.
         * @param  id  String value for id.
         * @param  amountMoney  Money value for amountMoney.
         */
        public Builder(String id, Money amountMoney) {
            this.id = id;
            this.amountMoney = amountMoney;
        }

        /**
         * Setter for id.
         * @param  id  String value for id.
         * @return Builder
         */
        public Builder id(String id) {
            this.id = id;
            return this;
        }

        /**
         * Setter for amountMoney.
         * @param  amountMoney  Money value for amountMoney.
         * @return Builder
         */
        public Builder amountMoney(Money amountMoney) {
            this.amountMoney = amountMoney;
            return this;
        }

        /**
         * Setter for status.
         * @param  status  String value for status.
         * @return Builder
         */
        public Builder status(String status) {
            this.status = status;
            return this;
        }

        /**
         * Setter for locationId.
         * @param  locationId  String value for locationId.
         * @return Builder
         */
        public Builder locationId(String locationId) {
            this.locationId = locationId;
            return this;
        }

        /**
         * Setter for appFeeMoney.
         * @param  appFeeMoney  Money value for appFeeMoney.
         * @return Builder
         */
        public Builder appFeeMoney(Money appFeeMoney) {
            this.appFeeMoney = appFeeMoney;
            return this;
        }

        /**
         * Setter for processingFee.
         * @param  processingFee  List of ProcessingFee value for processingFee.
         * @return Builder
         */
        public Builder processingFee(List<ProcessingFee> processingFee) {
            this.processingFee = processingFee;
            return this;
        }

        /**
         * Setter for paymentId.
         * @param  paymentId  String value for paymentId.
         * @return Builder
         */
        public Builder paymentId(String paymentId) {
            this.paymentId = paymentId;
            return this;
        }

        /**
         * Setter for orderId.
         * @param  orderId  String value for orderId.
         * @return Builder
         */
        public Builder orderId(String orderId) {
            this.orderId = orderId;
            return this;
        }

        /**
         * Setter for reason.
         * @param  reason  String value for reason.
         * @return Builder
         */
        public Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        /**
         * Setter for createdAt.
         * @param  createdAt  String value for createdAt.
         * @return Builder
         */
        public Builder createdAt(String createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        /**
         * Setter for updatedAt.
         * @param  updatedAt  String value for updatedAt.
         * @return Builder
         */
        public Builder updatedAt(String updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        /**
         * Setter for teamMemberId.
         * @param  teamMemberId  String value for teamMemberId.
         * @return Builder
         */
        public Builder teamMemberId(String teamMemberId) {
            this.teamMemberId = teamMemberId;
            return this;
        }

        /**
         * Builds a new {@link PaymentRefund} object using the set fields.
         * @return {@link PaymentRefund}
         */
        public PaymentRefund build() {
            return new PaymentRefund(id, amountMoney, status, locationId, appFeeMoney,
                    processingFee, paymentId, orderId, reason, createdAt, updatedAt, teamMemberId);
        }
    }
}
