/*
 * Copyright 2010-2012 Craig S. Dickson (http://craigsdickson.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.squeakysand.osgi.framework;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.squeakysand.commons.lang.NullParameterException;

/**
 * Simple {@link BundleActivator} that logs start and stop events, and holds on to a reference to the {@link BundleContext} object.
 * 
 * @author <a href="http://craigsdickson.com">Craig S. Dickson</a>
 */
public class BasicBundleActivator implements BundleActivator {

    private static final Logger LOG = LoggerFactory.getLogger(BasicBundleActivator.class);

    private Logger logger;
    private BundleContext context;

    /**
     * Creates a {@link BasicBundleActivator} instance that logs using its own logger.
     */
    public BasicBundleActivator() {
        this(LOG);
    }

    /**
     * Creates a {@link BasicBundleActivator} instance that logs using the supplied logger.
     * 
     * @param logger the {@link Logger} instance to use, cannot be <code>null</code>.
     */
    public BasicBundleActivator(Logger logger) {
        if (logger == null) {
            throw new NullParameterException("logger");
        }
        this.logger = logger;
    }

    @Override
    public void start(BundleContext context) throws Exception {
        this.context = context;
        logger.info(context.getBundle().getSymbolicName() + " started");
    }

    @Override
    public void stop(BundleContext context) throws Exception {
        this.context = null;
        logger.info(context.getBundle().getSymbolicName() + " stopped");
    }

    /**
     * Accessor for the {@link BundleContext} object associated with this {@link BundleActivator}.
     * 
     * @return the {@link BundleContext} object supplied when the {@link #start(BundleContext)} method was called.
     */
    public BundleContext getContext() {
        return context;
    }

}
