/*
 * Copyright 2010-2012 Craig S. Dickson (http://craigsdickson.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.squeakysand.osgi.framework;

import org.osgi.framework.BundleEvent;
import org.osgi.framework.BundleListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.squeakysand.commons.util.Predicate;

/**
 * {@link BundleListener} implementation that will listen for events based on the supplied event filter and will log them using SLF4J.
 * 
 * @author <a href="http://craigsdickson.com">Craig S. Dickson</a>
 */
public class LoggingBundleListener implements BundleListener {

    private static final Logger LOG = LoggerFactory.getLogger(LoggingBundleListener.class);

    private Logger logger;
    private Predicate<? super BundleEvent> eventFilter;

    /**
     * Creates a {@link LoggingBundleListener} that logs with its own logger.
     */
    public LoggingBundleListener() {
        this(LOG);
    }

    /**
     * Creates a {@link LoggingBundleListener} that logs with the supplied {@link Logger} instance.
     * 
     * @param logger the {@link Logger} instance ot use.
     */
    public LoggingBundleListener(Logger logger) {
        this.logger = logger;
        setDefaultPredicate();
    }

    /**
     * Sets the filter to use to determine which {@link BundleEvent}s should be logged.
     * 
     * @param eventFilter the filter to use.
     */
    public void setEventFilter(Predicate<? super BundleEvent> eventFilter) {
        if (eventFilter == null) {
            setDefaultPredicate();
        } else {
            this.eventFilter = eventFilter;
        }
    }

    @Override
    public void bundleChanged(BundleEvent event) {
        if (eventFilter.evaluate(event)) {
            logger.info(event.toString());
        }
    }

    private void setDefaultPredicate() {
        eventFilter = Predicate.ALWAYS_TRUE;
    }

}
