/*
 * Decompiled with CFR 0.152.
 */
package com.sourceclear.pysonar;

import com.google.common.base.Joiner;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.sourceclear.pysonar.Utils;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PythonProcesses {
    private static final int FILE_WATCH_TIMEOUT = 10000;
    private static final TypeToken<Map<String, Object>> JSON_RETURN_TYPE = new TypeToken<Map<String, Object>>(){};
    private static final Logger LOGGER = LoggerFactory.getLogger(PythonProcesses.class);
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Set<Version> pythonVersionsOnMachine = Collections.synchronizedSet(new HashSet(Version.values().length));
    private final Path parserScript;
    private final Path parserOutput;
    private final Path markerFile;
    private final Path parserLog;
    private Map<Version, Process> processes;

    public PythonProcesses(Path parserScript, Path parserOutput, Path markerFile, Path parserLog) {
        this.parserScript = parserScript;
        this.parserOutput = parserOutput;
        this.parserLog = parserLog;
        this.markerFile = markerFile;
        this.processes = new HashMap<Version, Process>();
    }

    public void ensurePythonIsAvailable() throws IOException {
        Set<Version> versionsOnMachine = PythonProcesses.getPythonVersionsOnMachine();
        if (versionsOnMachine.isEmpty()) {
            throw new IOException("Both python and python3 executables are not found on path");
        }
        if (versionsOnMachine.contains((Object)Version.PYTHON2)) {
            LOGGER.trace("{} available", (Object)Version.PYTHON2);
        } else {
            LOGGER.debug("python2 is not found, python2.X scans may fail.");
        }
        if (versionsOnMachine.contains((Object)Version.PYTHON3)) {
            LOGGER.trace("{} available", (Object)Version.PYTHON3);
        } else {
            LOGGER.debug("{} is not found, {} scans may fail.", (Object)Version.PYTHON3, (Object)Version.PYTHON3);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Set<Version> getPythonVersionsOnMachine() {
        Set<Version> set = pythonVersionsOnMachine;
        synchronized (set) {
            if (pythonVersionsOnMachine.isEmpty()) {
                for (Version version : Version.values()) {
                    if (!PythonProcesses.isPythonVersionAvailable(version)) continue;
                    pythonVersionsOnMachine.add(version);
                }
            }
        }
        return pythonVersionsOnMachine;
    }

    boolean isPythonAvailable(Version version) {
        return PythonProcesses.getPythonVersionsOnMachine().contains((Object)version);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean isPythonVersionAvailable(Version version) {
        Process p = null;
        ProcessBuilder builder = new ProcessBuilder(version.toString(), "-c", "print('hello')");
        builder.redirectErrorStream(true);
        try {
            p = builder.start();
            boolean bl = IOUtils.toString((InputStream)p.getInputStream(), (Charset)Utils.UTF_8).trim().equals("hello");
            return bl;
        }
        catch (IOException e) {
            boolean bl = false;
            return bl;
        }
        finally {
            if (p != null) {
                PythonProcesses.terminate(p);
            }
        }
    }

    public void start() throws IOException {
        if (!this.processes.isEmpty()) {
            throw new IllegalStateException("start should be called only once per Analyzer instance");
        }
        Set<Version> versionsOnMachine = PythonProcesses.getPythonVersionsOnMachine();
        if (versionsOnMachine.isEmpty()) {
            throw new IOException("Both python and python3 executables are not found on path");
        }
        for (Version version : versionsOnMachine) {
            this.processes.put(version, this.startPythonProcess(version));
        }
    }

    public void cleanup() {
        if (this.processes.isEmpty()) {
            LOGGER.warn("cleanup should be called only when there are processes");
            return;
        }
        for (Version version : Version.values()) {
            if (!this.processes.containsKey((Object)version)) continue;
            PythonProcesses.terminate(this.processes.get((Object)version));
        }
    }

    private Process getPythonProcess(Version version) throws IOException {
        Process process = this.processes.get((Object)version);
        Objects.requireNonNull(process);
        if (!process.isAlive()) {
            LOGGER.warn("{} was killed, recreating", (Object)version);
            process = this.startPythonProcess(version);
            this.processes.put(version, process);
        }
        return process;
    }

    @NotNull
    private Process startPythonProcess(Version python) throws IOException {
        ProcessBuilder builder = new ProcessBuilder(python.toString(), "-i", this.parserScript.toAbsolutePath().toString());
        builder.redirectErrorStream(true);
        builder.redirectOutput(this.parserLog.toFile());
        builder.environment().remove("PYTHONPATH");
        try {
            return builder.start();
        }
        catch (IOException e) {
            String fullCommand = Joiner.on((String)" ").join(builder.command());
            throw new IOException("Could not run command " + fullCommand, e);
        }
    }

    @NotNull
    public Map<String, Object> parseFile(Path filename, Version version) throws IOException {
        String a1 = Utils.escapeWindowsPath(filename.toAbsolutePath().toString());
        String a2 = Utils.escapeWindowsPath(this.parserOutput.toAbsolutePath().toString());
        String a3 = Utils.escapeWindowsPath(this.markerFile.toAbsolutePath().toString());
        String dumpCommand = String.format("parse_dump('%s', '%s', '%s')", a1, a2, a3);
        Process python = this.getPythonProcess(version);
        Objects.requireNonNull(python);
        if (!this.sendCommand(dumpCommand, python)) {
            throw new IOException("Could not send command " + dumpCommand + " to" + (Object)((Object)version));
        }
        this.waitForMarkerFile();
        String json = Utils.readFile(this.parserOutput);
        Map parsed = (Map)GSON.fromJson(json, JSON_RETURN_TYPE.getType());
        if (parsed != null) {
            return parsed;
        }
        throw new IOException("Could not read file " + this.parserOutput);
    }

    private void waitForMarkerFile() throws IOException {
        long waitStart = System.currentTimeMillis();
        while (!Files.exists(this.markerFile, new LinkOption[0])) {
            if (System.currentTimeMillis() - waitStart > 10000L) {
                throw new IOException("Could not parse file after 10s");
            }
            try {
                Thread.sleep(1L);
            }
            catch (InterruptedException e) {
                throw new IOException("Interrupted while waiting for file");
            }
        }
    }

    private boolean sendCommand(String command, @NotNull Process pythonProcess) {
        try {
            OutputStreamWriter writer = new OutputStreamWriter(pythonProcess.getOutputStream());
            writer.write(command);
            writer.write("\n");
            writer.flush();
            return true;
        }
        catch (IOException e) {
            LOGGER.error("Failed to send command to interpreter: {}", (Object)command, (Object)e);
            return false;
        }
    }

    private static void terminate(Process process) {
        process.destroy();
        try {
            LOGGER.trace("Python process killed {}", (Object)process.waitFor());
        }
        catch (InterruptedException e) {
            LOGGER.error("Process was not killed", (Throwable)e);
        }
    }

    public static enum Version {
        PYTHON2("python"),
        PYTHON3("python3");

        private final String name;

        private Version(String s) {
            this.name = s;
        }

        public String toString() {
            return this.name;
        }
    }
}

