/*
 * © Copyright 2015 -  SourceClear Inc
 */

package com.srcclr.sdk;

import com.fasterxml.jackson.annotation.JsonSetter;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import java.util.Objects;

/**
 *
 */
@JsonDeserialize(builder=Coords.Builder.class)
public class Coords {

  public static class Builder {

    private CoordinateType coordinateType;

    private String coordinate1;

    private String coordinate2;

    private String version;

    private String scope;

    private String platform;

    private String commitHash;

    /**
     * This is only used for backwards compatibility with older serialized code and should not be called by anyone
     * except Jackson.  The MatchQuery API also uses the older 'buildType' value.
     */
    @Deprecated
    @JsonSetter("buildType")
    public Builder withBuildType(BuildType buildType) {

      //
      // TODO: Once depy generation 2 lands we should be able to remove this.
      //
      // Our current depy code has a hard-coded value of PIP for buildType,
      // but that's not a valid coordinate type.
      //
      coordinateType = buildType == BuildType.PIP ? CoordinateType.PYPI : CoordinateType.toCoordType(buildType.name());

      return this;
    }

    public Builder withCoordinateType(CoordinateType coordinateType) {
      this.coordinateType = coordinateType;
      return this;
    }

    public Builder withCoordinate1(String coordinate1) {
      this.coordinate1 = coordinate1;
      return this;
    }

    public Builder withCoordinate2(String coordinate2) {
      this.coordinate2 = coordinate2;
      return this;
    }

    public Builder withCoordinates(String... coords) {

      int len = coords.length;

      if (len < 1 || len > 2) {
        throw new IllegalArgumentException("Coords must be 1 or 2 in length");
      }

      coordinate1 = coords[0];
      if (len == 2) {
        coordinate2 = coords[1];
      }

      return this;
    }

    public Builder withVersion(String version) {
      this.version = version;
      return this;
    }

    public Builder withScope(String scope) {
      this.scope = scope;
      return this;
    }

    public Builder withPlatform(String platform) {
      this.platform = platform;
      return this;
    }

    public Builder withCommitHash(String commitHash) {
      this.commitHash = commitHash;
      return this;
    }

    public Coords build() {
      return new Coords(this);
    }
  }

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /**
   * Standard value to be used for unknown coordinate element.
   */
  public final static String UNDEFINED_VALUE = "<undefined>";

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  @NotNull
  private final CoordinateType coordinateType;

  @NotNull
  @Size(min = 1)
  private final String coordinate1;

  private final String coordinate2;

  @NotNull
  @Size(min = 1)
  private final String version;

  private final String scope;

  private final String platform;

  private final String commitHash;

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private Coords(Builder builder) {
    coordinateType = builder.coordinateType;
    coordinate1 = builder.coordinate1;
    coordinate2 = builder.coordinate2;
    version = builder.version;
    scope = builder.scope;
    platform = builder.platform;
    commitHash = builder.commitHash;
  }

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  public String toRawString() {
    String str = this.coordinate2 != null ?
      String.format("%s:%s:%s", this.coordinate1, this.coordinate2,
        this.version) : String.format("%s:%s", this.coordinate1, this.version);

    if ((scope != null) && !scope.isEmpty()) {
      str += ":" + scope;
    }

    if (commitHash != null) {
      str += ":" + commitHash;
    }

    return str;
  }

  //------------------------ Implements:

  //------------------------ Overrides: Object

  @Override
  public String toString() {
    String str = this.coordinate2 != null ?
      String.format("%s/%s:%s:%s", this.coordinateType, this.coordinate1,
        this.coordinate2, this.version) : String.format("%s/%s:%s",
      this.coordinateType, this.coordinate1, this.version);


    if ((scope != null) && !scope.isEmpty()) {
      str += ":" + scope;
    }

    if (commitHash != null) {
      str += ":" + commitHash;
    }

    return str;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof Coords)) return false;
    Coords that = (Coords) o;
    return Objects.equals(getCoordinateType(), that.getCoordinateType()) &&
      Objects.equals(getCoordinate1(), that.getCoordinate1()) &&
      Objects.equals(getCoordinate2(), that.getCoordinate2()) &&
      Objects.equals(getVersion(), that.getVersion()) &&
      Objects.equals(getScope(), that.getScope()) &&
      Objects.equals(getPlatform(), that.getPlatform()) &&
      Objects.equals(getCommitHash(), that.getCommitHash());
  }

  @Override
  public int hashCode() {
    return Objects.hash(getCoordinateType(), getCoordinate1(), getCoordinate2(), getVersion(), getScope(), getPlatform(), getCommitHash());
  }

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------

  public CoordinateType getCoordinateType() {
    return coordinateType;
  }

  public String getCoordinate1() {
    return coordinate1;
  }

  public String getCoordinate2() {
    return coordinate2;
  }

  public String getVersion() {
    return version;
  }

  public String getScope() {
    return scope;
  }

  public String getPlatform() {
    return platform;
  }

  public String getCommitHash() { return commitHash; }
}
