/*
 * © Copyright 2015 -  SourceClear Inc
 */

package com.srcclr.sdk;

import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;
import org.jheaps.annotations.VisibleForTesting;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.validation.Valid;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;

/**
 * Immutable data class which can easily be serialized via JSON. This allows us to pass dependency graphs between
 * separate JVM processes.
 *
 * This class utilizes the builder pattern, but has default annotations to allow Jackson to deserialize without further
 * configuration.
 */
@JsonIdentityInfo(generator= ObjectIdGenerators.IntSequenceGenerator.class)
@JsonIgnoreProperties(ignoreUnknown = true)
public class LibraryGraph {

  public static class Builder {

    private String moduleName;

    private Coords coords;

    private Set<LibraryGraph> directs = new LinkedHashSet<>();

    private String filename;

    private Integer lineNumber;

    private String sha1;

    private String sha2;

    private String bytecodeHash;

    public Builder withCoords(Coords coords) {
      this.coords = coords;
      return this;
    }

    public Builder withDirect(LibraryGraph direct) {
      directs.add(direct);
      return this;
    }

    public Builder withDirects(Set<LibraryGraph> directs) {
      this.directs = new HashSet<>(directs);
      return this;
    }

    public Builder withFilename(String filename) {
      this.filename = filename;
      return this;
    }

    public Builder withLineNumber(Integer lineNumber) {
      this.lineNumber = lineNumber;
      return this;
    }

    public Builder withModuleName(String moduleName) {
      this.moduleName = moduleName;
      return this;
    }

    public Builder withSha1(String sha1) {
      this.sha1 = sha1;
      return this;
    }

    public Builder withSha2(String sha2) {
      this.sha2 = sha2;
      return this;
    }

    public Builder withBytecodeHash(String bytecodeHash) {
      this.bytecodeHash = bytecodeHash;
      return this;
    }

    @Override
    public int hashCode() {
      return Objects.hash(coords, filename, lineNumber);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      return Objects.equals(hashCode(), o.hashCode());
    }

    public LibraryGraph build() {
      return new LibraryGraph(this);
    }
  }

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  @Valid
  private Coords coords;

  @Valid
  private Set<LibraryGraph> directs;

  private String filename;

  private Integer lineNumber;

  private String moduleName;

  private String sha1;

  private String sha2;

  private String bytecodeHash;

  @JsonIgnore
  private int hashCache;

  @JsonIgnore
  private boolean isHashSet = false;
  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\  

  @SuppressWarnings("unused")
  public LibraryGraph() {
  }

  private LibraryGraph(Builder builder) {
    coords = builder.coords;
    directs = builder.directs;
    filename = builder.filename;
    lineNumber = builder.lineNumber;
    moduleName = builder.moduleName;
    sha1 = builder.sha1;
    sha2 = builder.sha2;
    bytecodeHash = builder.bytecodeHash;
  }

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /**
   * @return Compute the total size (count) of this LibraryGraph (or sub graph), including this component and all descendants.
   */
  public int size() {
    int size = 1;

    for (LibraryGraph n : directs) {
      size += n.size();
    }

    return size;
  }

  //------------------------ Implements:

  //------------------------ Overrides:


  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    LibraryGraph that = (LibraryGraph) o;

    return Objects.equals(coords, that.coords) &&
           Objects.equals(directs, that.directs) &&
           Objects.equals(filename, that.filename) &&
           Objects.equals(lineNumber, that.lineNumber);
  }

  @Override
  public int hashCode() {
    // we may encounter DEEP hashes
    // since every attribute is immutable after building, it should be safe to just cache this hash
    if(!isHashSet){
      hashCache = Objects.hash(coords, directs, filename, lineNumber);
      isHashSet = true;
    }
    return hashCache;
  }


  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------


  /**
   * @return The Coords that define this component.
   */
  public Coords getCoords() {
    return coords;
  }

  /**
   * @return An unmodifiable Set of child components that this component relies on.
   */
  public Set<LibraryGraph> getDirects() {
    return directs;
  }

  /**
   * The file name (plus relative path) of where this dependency originated. This only applies for direct dependencies.
   *
   * @return The source file, or null if this dependency is transitive or otherwise couldn't be computed.
   */
  @Nullable
  public String getFilename() {
    return filename;
  }

  /**
   * The line number of where this dependency originated. This only applies for direct dependencies.
   *
   * @return The line number, or null if this dependency is transitive or otherwise couldn't be computed.
   */
  @Nullable
  public Integer getLineNumber() {
    return lineNumber;
  }

  /**
   * The module name denotes which sub-project (module) this graph belongs to.  This is needed because often there's
   * no indication from the graph itself which module it derives from.
   */
  @Nullable
  public String getModuleName() {return moduleName;}

  /**
   * The SHA-1 of the library, if collected.
   */
  @Nullable
  public String getSha1() {
    return sha1;
  }

  /**
   * The SHA-256 of the library, if collected.
   */
  @Nullable
  public String getSha2() {
    return sha2;
  }

  /**
   * The bytecode hash, if the library is a Jar and it was collected.
   * @return
   */
  @Nullable
  public String getBytecodeHash() {
    return bytecodeHash;
  }

  /**
   * This method is added in order to deliberately create circular dependencies
   * for the purpose of testing. It should ONLY be used in unit test code.
   * We marked it DEPRECATED to alarm developer not use it in production code.
   * Please do NOT really DEPRECATE it, it is used in UTILS project to test circular
   * dependency paths.
   *
   * @param direct
   */
  @VisibleForTesting
  @Deprecated
  public void addDirect(LibraryGraph direct) {
    directs.add(direct);
  }

  /**
   * This method is added in order to deliberately create circular dependencies
   * for the purpose of testing. It should ONLY be used in unit test code.
   * We marked it DEPRECATED to alarm developer not use it in production code.
   * Please do NOT really DEPRECATE it, it is used in UTILS project to test circular
   * dependency paths.
   *
   * @param direct
   */
  @VisibleForTesting
  @Deprecated
  public void removeDirect(LibraryGraph direct) {
    directs.remove(direct);
  }
}
