/*
 * © Copyright 2015 -  SourceClear Inc
 */

package com.srcclr.sdk;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

import javax.annotation.concurrent.Immutable;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * This class acts as a simple container for a collection of LibraryGraphs.  It's designed to be both immutable and
 * serializable so it can be used as handoff data between processes running on different JVMs.
 */
@Immutable
@JsonDeserialize(using = LibraryGraphContainerDeserializer.class)
public class LibraryGraphContainer {

  public static class Builder {

    private Set<LibraryGraph> graphs = new HashSet<>();
    private DepGraph depGraph;

    public Builder withGraph(LibraryGraph graph) {
      graphs.add(graph);
      return this;
    }

    public Builder withGraphs(Collection<LibraryGraph> graphs) {
      this.graphs.addAll(graphs);
      return this;
    }

    public Builder withDepGraph(DepGraph depGraph) {
      this.depGraph = depGraph;
      return this;
    }

    public LibraryGraphContainer build() {
      return new LibraryGraphContainer(this);
    }
  }

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  public static final LibraryGraphContainer EMPTY_GRAPH = new Builder().build();

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  @NotNull
  @Valid
  @Deprecated
  protected final Set<LibraryGraph> graphs;

  @JsonIgnore
  public DepGraph depGraph;

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\  

  protected LibraryGraphContainer(Builder builder) {
    this.graphs = Collections.unmodifiableSet(builder.graphs);
    this.depGraph = builder.depGraph;
  }

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //------------------------ Implements:

  //------------------------ Overrides:

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------

  /**
   * Get the generation of serialization format that this container is compatible with. This is actually on the
   * LibraryGraphSerializer object, but we put it here so that Jackson places it in the serialized data.
   * @return generation
   */
  public long getGeneration() {
    return LibraryGraphSerializer.getCurrentGeneration();
  }

  /**
   * Get the unmodifiable Set of ComponentGraphs.
   */
  public Set<LibraryGraph> getGraphs() {
    return graphs;
  }

  public DepGraph getDepGraph() {
    return depGraph;
  }
}
