/*
 * Copyright (c) 2016 -  SourceClear Inc
 */

package com.srcclr.sdk;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * A Record is a specific response to an individual query of some type.  For example, if multiple coordinates are looked
 * up in a single API call, one Report will be returned with one record per coordinate.  Each Record is fully
 * independent of any others.  All data is self-contained, it does not reference any data outside of itself.
 *
 * A record may contain Graphs, Components, and Vulnerabilities.  It may also be empty if no match could be made.
 */
@Immutable
@JsonDeserialize(builder=Record.Builder.class)
public class Record {

  public static class Builder {

    private RecordMetadata metadata;

    private Collection<LibraryGraph> graphs = new LinkedList<>();

    private Collection<Library> libraries = new LinkedList<>();

    private Collection<Vulnerability> vulnerabilities = new LinkedList<>();

    public Builder withMetadata(RecordMetadata metadata) {
      this.metadata = metadata;
      return this;
    }

    public Builder withGraphs(Collection<LibraryGraph> graphs) {
      this.graphs = new ArrayList<>(graphs);
      return this;
    }

    public Builder withGraph(LibraryGraph graph) {
      graphs.add(graph);
      return this;
    }

    public Builder withLibraries(Collection<Library> libraries) {
      this.libraries = new ArrayList<>(libraries);
      return this;
    }

    public Builder withVulnerabilities(Collection<Vulnerability> vulnerabilities) {
      this.vulnerabilities = new ArrayList<>(vulnerabilities);
      return this;
    }

    public Record build() {
      return new Record(this);
    }

  }

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private final RecordMetadata metadata;

  private final Collection<LibraryGraph> graphs;

  private final Collection<Library> libraries;

  private final Collection<Vulnerability> vulnerabilities;

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private Record(Builder builder) {
    metadata = builder.metadata;
    graphs = Collections.unmodifiableCollection(builder.graphs);
    libraries = Collections.unmodifiableCollection(builder.libraries);
    vulnerabilities = Collections.unmodifiableCollection(builder.vulnerabilities);
  }
  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //------------------------ Implements:

  //------------------------ Overrides:

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------

  @Nonnull
  public RecordMetadata getMetadata() {
    return metadata;
  }

  @Nonnull
  public Collection<LibraryGraph> getGraphs() {
    return graphs;
  }

  @Nonnull
  public Collection<Library> getLibraries() {
    return libraries;
  }

  @Nonnull
  public Collection<Vulnerability> getVulnerabilities() {
    return vulnerabilities;
  }

}
