/*
 * Copyright (c) 2016 -  SourceClear Inc
 */

package com.srcclr.sdk;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * The top-level object that defines a report for some type of query.  This may be
 * coordinate lookups (either singular or plural), scans results, or custom report
 * output from the SourceClear platform.
 *
 * Each Report instance will contain one or more records, where each record is
 * a specific response to a query.  In the case of a single coordinate lookup,
 * the Report would contain 1 record.  For n-coordinates, we would return n-records.
 *
 */
@Immutable
@JsonDeserialize(builder=Report.Builder.class)
public class Report {

  public static class Builder {

    private ReportMetadata metadata = new ReportMetadata.Builder().build();

    private List<Record> records = new LinkedList<>();

    public Builder withMetadata(ReportMetadata metadata) {
      this.metadata = metadata;
      return this;
    }

    public Builder withRecords(List<Record> records) {
      this.records = new ArrayList<>(records);
      return this;
    }

    public Report build() {
      return new Report(this);
    }
  }

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private final ReportMetadata metadata;

  private final List<Record> records;

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\  

  private Report(Builder builder) {
    this.metadata = builder.metadata;
    this.records = Collections.unmodifiableList(builder.records);
  }

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //------------------------ Implements:

  //------------------------ Overrides:

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------

  @Nonnull
  public ReportMetadata getMetadata() {
    return metadata;
  }

  @Nonnull
  public List<Record> getRecords() {
    return records;
  }
}
