/*
 * Copyright (c) 2016 -  SourceClear Inc
 */

package com.srcclr.sdk;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;

import javax.annotation.concurrent.Immutable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;

/**
 *
 */
@Immutable
@JsonDeserialize(builder=Vulnerability.Builder.class)
public class Vulnerability extends BaseModel {

  public static class Builder extends BaseBuilder<Builder, Vulnerability> {

    private Date disclosureDate;

    private boolean hasExploits;

    private String cve;

    private String title;

    private String overview;

    private LanguageType language;

    private Set<String> vulnerabilityTypes = new HashSet<>();

    private Float cvssScore;

    private Float cvss3Score;

    private Collection<VulnerabilityLibrary> libraries = new LinkedList<>();

    public Builder withDisclosureDate(Date disclosureDate) {
      this.disclosureDate = disclosureDate;
      return this;
    }

    public Builder withHasExploits(boolean hasExploits) {
      this.hasExploits = hasExploits;
      return this;
    }

    public Builder withCve(String cve) {
      this.cve = cve;
      return this;
    }

    public Builder withTitle(String title) {
      this.title = title;
      return this;
    }

    public Builder withOverview(String overview) {
      this.overview = overview;
      return this;
    }

    public Builder withLanguage(LanguageType language) {
      this.language = language;
      return this;
    }

    public Builder withVulnerabilityTypes(Set<String> vulnerabilityTypes) {
      this.vulnerabilityTypes = vulnerabilityTypes;
      return this;
    }

    public Builder withCvssScore(Float cvssScore) {
      this.cvssScore = cvssScore;
      return this;
    }

    public Builder withCvss3Score(Float cvss3Score) {
      this.cvss3Score = cvss3Score;
      return this;
    }

    public Builder withLibraries(Collection<VulnerabilityLibrary> libraries) {
      this.libraries = new ArrayList<>(libraries);
      return this;
    }

    public Vulnerability build() {
      return new Vulnerability(this);
    }

  }

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private final Date disclosureDate;

  private final boolean hasExploits;

  private final String cve;

  private final String title;

  private final String overview;

  private final LanguageType language;

  private final Set<String> vulnerabilityTypes;

  private final Float cvssScore;

  private final Float cvss3Score;

  private final Collection<VulnerabilityLibrary> libraries;

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\  

  private Vulnerability(Builder builder) {
    super(builder);
    disclosureDate = builder.disclosureDate;
    hasExploits = builder.hasExploits;
    cve = builder.cve;
    title = builder.title;
    overview = builder.overview;
    language = builder.language;
    vulnerabilityTypes = builder.vulnerabilityTypes;
    cvssScore = builder.cvssScore;
    cvss3Score = builder.cvss3Score;
    libraries = Collections.unmodifiableCollection(builder.libraries);
  }

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //------------------------ Implements:

  //------------------------ Overrides:

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------

  public Date getDisclosureDate() {
    return disclosureDate;
  }

  @JsonProperty("hasExploits")
  public boolean hasExploits() {
    return hasExploits;
  }

  public String getCve() {
    return cve;
  }

  public String getTitle() {
    return title;
  }

  public String getOverview() {
    return overview;
  }

  public LanguageType getLanguage() {
    return language;
  }

  public Set<String> getVulnerabilityTypes() {
    return vulnerabilityTypes;
  }

  public Float getCvssScore() {
    return cvssScore;
  }

  public Float getCvss3Score() {
    return cvss3Score;
  }

  public Collection<VulnerabilityLibrary> getLibraries() {
    return libraries;
  }
}
